/*
    lio_serial.c
    serial line access

    This file is part of the LowIO library project
    Copyright (C) 2001 by Stephan Linz <linz@li-pro.net>

    Some stuff is comming from the MUSCLE Towitoko driver project
    Copyright (C) 2000 by Carlos Prados <cprados@yahoo.com>

    This library is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Bosten, MA 02111-1307 USA
*/

#define _LIO_SERIAL_C_ 1


#include "system.h"


/*
 * not exported macros definition
 */

#define BITRATE_TO_TIO(A)	((A) >=       0) && ((A) <      50) ? B0       : \
				((A) >=      50) && ((A) <      75) ? B50      : \
				((A) >=      75) && ((A) <     110) ? B75      : \
				((A) >=     110) && ((A) <     134) ? B110     : \
				((A) >=     134) && ((A) <     150) ? B134     : \
				((A) >=     150) && ((A) <     200) ? B150     : \
				((A) >=     200) && ((A) <     300) ? B200     : \
				((A) >=     300) && ((A) <     600) ? B300     : \
				((A) >=     600) && ((A) <    1200) ? B600     : \
				((A) >=    1200) && ((A) <    1800) ? B1200    : \
				((A) >=    1800) && ((A) <    2400) ? B1800    : \
				((A) >=    2400) && ((A) <    4800) ? B2400    : \
				((A) >=    4800) && ((A) <    9600) ? B4800    : \
				((A) >=    9600) && ((A) <   19200) ? B9600    : \
				((A) >=   19200) && ((A) <   38400) ? B19200   : \
				((A) >=   38400) && ((A) <   57600) ? B38400   : \
				((A) >=   57600) && ((A) <  115200) ? B57600   : \
				((A) >=  115200) && ((A) <  230400) ? B115200  : \
				((A) >=  230400) && ((A) <  460800) ? B230400  : \
				((A) >=  460800) && ((A) <  500000) ? B460800  : \
				((A) >=  500000) && ((A) <  576000) ? B500000  : \
				((A) >=  576000) && ((A) <  921600) ? B576000  : \
				((A) >=  921600) && ((A) < 1000000) ? B921600  : \
				((A) >= 1000000) && ((A) < 1152000) ? B1000000 : \
				((A) >= 1152000) && ((A) < 1500000) ? B1152000 : \
				((A) >= 1500000) && ((A) < 2000000) ? B1500000 : \
				((A) >= 2000000) && ((A) < 2500000) ? B2000000 : \
				((A) >= 2500000) && ((A) < 3000000) ? B2500000 : \
				((A) >= 3000000) && ((A) < 3500000) ? B3000000 : \
				((A) >= 3500000) && ((A) < 4000000) ? B3500000 : B4000000

#define BITRATE_TO_INT(A)	(int)( \
				    (A) == B0       ? 0       : \
				    (A) == B50      ? 50      : \
				    (A) == B75      ? 75      : \
				    (A) == B110     ? 110     : \
				    (A) == B134     ? 134     : \
				    (A) == B150     ? 150     : \
				    (A) == B200     ? 200     : \
				    (A) == B300     ? 300     : \
				    (A) == B600     ? 600     : \
				    (A) == B1200    ? 1200    : \
				    (A) == B1800    ? 1800    : \
				    (A) == B2400    ? 2400    : \
				    (A) == B4800    ? 4800    : \
				    (A) == B9600    ? 9600    : \
				    (A) == B19200   ? 19200   : \
				    (A) == B38400   ? 38400   : \
				    (A) == B57600   ? 57600   : \
				    (A) == B115200  ? 115200  : \
				    (A) == B230400  ? 230400  : \
				    (A) == B460800  ? 460800  : \
				    (A) == B500000  ? 500000  : \
				    (A) == B576000  ? 576000  : \
				    (A) == B921600  ? 921600  : \
				    (A) == B1000000 ? 1000000 : \
				    (A) == B1152000 ? 1152000 : \
				    (A) == B1500000 ? 1500000 : \
				    (A) == B2000000 ? 2000000 : \
				    (A) == B2500000 ? 2500000 : \
				    (A) == B3000000 ? 3000000 : \
				    (A) == B3500000 ? 3500000 : \
				    (A) == B4000000 ? 4000000 : -1 \
				)

#define CHARSIZE_TO_TIO(A)	(A) == 5 ? CS5 : \
				(A) == 6 ? CS6 : \
				(A) == 7 ? CS7 : \
				(A) == 8 ? CS8 : 0

#define CHARSIZE_TO_INT(A)	(int)( \
				    ((A) & CSIZE) == CS5 ? 5 : \
				    ((A) & CSIZE) == CS6 ? 6 : \
				    ((A) & CSIZE) == CS7 ? 7 : \
				    ((A) & CSIZE) == CS8 ? 8 : -1 \
				)


/*
 * internal functions declaration
 */
static int LIOS_CanRead __P((int fd, unsigned delay_ms, unsigned timeout_ms));
static int LIOS_CanWrite __P((int fd, unsigned delay_ms, unsigned timeout_ms));
static void LIOS_DeviceName __P((int port, unsigned char * filename, int length));
static void LIOS_Clear __P((LIO_Serial * io));





/*
 * public functions definition
 */

LIO_Serial * LIO_Serial_New(void)
{
    LIO_Serial *	io;

    io = (LIO_Serial *)malloc(sizeof(LIO_Serial));

    if (io != NULL) {
	LIOS_Clear(io);
    }

    memset(io, 0, sizeof(LIO_Serial));

    DBG_INFO("have new io structure");

    return io;
}



int LIO_Serial_Init(LIO_Serial * io, int port)
{
    unsigned char		filename[BUFSIZ] = "\0";
    LIO_Serial_Transfermode	trm;

    if ( (port < 1) || (port > 128) ) {
	DBG_ERROR("port number %i out of range", port);
	return 0;
    }

    LIOS_DeviceName(port, filename, BUFSIZ);

    io->port  = port;
    io->fd    = open(filename, O_RDWR | O_NOCTTY);

    if (io->fd < 0) {
	DBG_ERROR("can't open device \"%s\"", filename);
	return 0;
    }

    trm = LIO_SERIAL_TRM_BYTEWISE;
    if (!LIO_Serial_SetAM(io, &trm)) {
	DBG_ERROR("can't set default transfer mode to %s(%i)",
	    LIO_SERIAL_TRM_VALSTR(trm), trm);
	return 0;
    }

    DBG_INFO("have io structure connected to device \"%s\"", filename);
    return 1;
}



int LIO_Serial_GetProp(LIO_Serial * io, LIO_Serial_Properties * prop)
{
    struct termios	tio;
    speed_t		i_speed, o_speed;
    unsigned int	ctl;

    /* check the io/properties anchor */
    if ((io == NULL) || (prop == NULL)) {
	DBG_ERROR("missing pointer to either io or prop structure");
	return 0;
    }

    /* fill out my termios */
    if (tcgetattr(io->fd, &tio) < 0) {
	DBG_ERROR("can't get current termios on io->fd : %i", io->fd);
	return 0; /* TODO: check errno */
    }

    /* get and check input/output bitrate */
    o_speed = cfgetospeed(&tio);
    prop->bitrate.output = BITRATE_TO_INT(o_speed);

    i_speed = cfgetispeed(&tio);
    prop->bitrate.input = BITRATE_TO_INT(i_speed);

    if ( (prop->bitrate.output < 0) || (prop->bitrate.input < 0) ) {
	DBG_ERROR("found unknown i/o baud rate inside of current termios");
	return 0;
    }

    DBG_INFO("found i/o baud rate : %i/%i",
	prop->bitrate.input, prop->bitrate.output);

    /* get and check character size (bits) */
    prop->parameter.bits = CHARSIZE_TO_INT(tio.c_cflag);
    if (prop->parameter.bits < 0) {
	DBG_ERROR("found unknown character size inside of current termios");
	return 0;
    }

    DBG_INFO("found character size : %i", prop->parameter.bits);

    /* get type of parity bit */
    if (ALL_BITS_SET(PARENB, tio.c_cflag)) {
	if (ALL_BITS_SET(PARODD, tio.c_cflag)) {

	    prop->parameter.parity = LIO_SERIAL_PAR_ODD;
	    DBG_INFO("found odd parity");

	} else {

	    prop->parameter.parity = LIO_SERIAL_PAR_EVEN;
	    DBG_INFO("found even parity");

	}
    } else {

	prop->parameter.parity = LIO_SERIAL_PAR_NONE;
	DBG_INFO("found none parity");

    }

    /* get number of stop bits */
    if (ALL_BITS_SET(CSTOPB, tio.c_cflag)) {

	prop->parameter.stopbits = 2;

    } else {

	prop->parameter.stopbits = 1;

    }

    DBG_INFO("found %i stopbits", prop->parameter.stopbits);

    /* get and check the modem lines */
    if (ioctl(io->fd, TIOCMGET, &ctl) < 0) {
	DBG_ERROR("can't get current level of modem lines on io->fd : %i", io->fd);
	return 0; /* TODO: check errno */
    }

    prop->modemlines.dtr = ctl & TIOCM_DTR ? LIO_SERIAL_LM_HIGH : LIO_SERIAL_LM_LOW;
    prop->modemlines.rts = ctl & TIOCM_RTS ? LIO_SERIAL_LM_HIGH : LIO_SERIAL_LM_LOW;
    prop->modemlines.dsr = ctl & TIOCM_DSR ? LIO_SERIAL_LM_HIGH : LIO_SERIAL_LM_LOW;
    prop->modemlines.cts = ctl & TIOCM_CTS ? LIO_SERIAL_LM_HIGH : LIO_SERIAL_LM_LOW;
    prop->modemlines.dcd = ctl & TIOCM_CD  ? LIO_SERIAL_LM_HIGH : LIO_SERIAL_LM_LOW;
    prop->modemlines.ri  = ctl & TIOCM_RI  ? LIO_SERIAL_LM_HIGH : LIO_SERIAL_LM_LOW;

    return 1;
}



int LIO_Serial_SetProp(LIO_Serial * io, LIO_Serial_Properties * prop)
{
    struct termios	tio;
    unsigned int	ctl;

    /* check the io/properties anchor */
    if ((io == NULL) || (prop == NULL)) {
	DBG_ERROR("missing pointer to either io or prop structure");
	return 0;
    }

    /* get, change and set the out modem lines */
    if (ioctl(io->fd, TIOCMGET, &ctl) < 0) {
	DBG_ERROR("can't get current level of modem lines on io->fd : %i", io->fd);
	return 0; /* TODO: check errno */
    }

    ctl = prop->modemlines.dtr == LIO_SERIAL_LM_HIGH ?
	    ctl | TIOCM_DTR : ctl & ~TIOCM_DTR;
    ctl = prop->modemlines.rts == LIO_SERIAL_LM_HIGH ?
	    ctl | TIOCM_RTS : ctl & ~TIOCM_RTS;

    if (ioctl(io->fd, TIOCMSET, &ctl) < 0) {
	DBG_ERROR("can't set new level of modem lines on io->fd : %i", io->fd);
	return 0; /* TODO: check errno */
    }

    DBG_INFO("have setup level of modem lines DTR(%i) and RTS(%i)",
	prop->modemlines.dtr, prop->modemlines.rts);

    memset(&tio, 0, sizeof(tio));

    /* set input/output bitrate */
    cfsetispeed(&tio, BITRATE_TO_TIO(prop->bitrate.input));
    cfsetospeed(&tio, BITRATE_TO_TIO(prop->bitrate.output));

    DBG_INFO("have setup i/o baud rate to : %i/%i",
	prop->bitrate.input, prop->bitrate.output);

    /* set character size (bits) */
    tio.c_cflag |= CHARSIZE_TO_TIO(prop->parameter.bits);

    DBG_INFO("have setup character size to : %i", prop->parameter.bits);

    /* set type of parity bit -- default NONE */
    switch (prop->parameter.parity) {

      case LIO_SERIAL_PAR_ODD:
	tio.c_cflag |= (PARENB | PARODD); /* activate odd parity check/build */
	tio.c_iflag |= INPCK;		  /* activate parity check for input */
	DBG_INFO("have setup odd parity");
	break;

      case LIO_SERIAL_PAR_EVEN:
	tio.c_cflag &= (~PARODD);	  /* even parity */
	tio.c_cflag |= PARENB;		  /* activate parity check/build */
	tio.c_iflag |= INPCK;		  /* activate parity check for input */
	DBG_INFO("have setup even parity");
	break;

      case LIO_SERIAL_PAR_NONE:
	tio.c_cflag &= (~PARENB);	  /* disable parity check/build */
	tio.c_iflag &= (~INPCK);	  /* disable parity check for input */
	DBG_INFO("have setup none parity");
	break;
    }

    /* set number of stop bits */
    switch (prop->parameter.stopbits) {

      case 1:
	tio.c_cflag &= (~CSTOPB);
	break;

      case 2:
	tio.c_cflag |= CSTOPB;
	break;
    }

    DBG_INFO("have setup %i stopbits", prop->parameter.stopbits);

    /* TODO: what is it -- description ??? */
    tio.c_cflag		|= (CREAD | HUPCL | CLOCAL);
    tio.c_iflag		&= ~(IGNPAR | PARMRK | INLCR | ICRNL | ISTRIP);
    tio.c_iflag		|= BRKINT;
    tio.c_lflag		&= ~(ICANON | ECHO);
    tio.c_oflag		= 0;
    tio.c_lflag		= 0;
    tio.c_cc[VMIN]	= 1;
    tio.c_cc[VTIME]	= 0;

    /* activate my new termios */
    if (tcsetattr(io->fd, TCSANOW, &tio) < 0) {
	DBG_ERROR("can't set new termios on io->fd : %i", io->fd);
	return 0; /* TODO: check errno */
    }

    if (tcflush(io->fd, TCIFLUSH) < 0) {
	DBG_ERROR("can't flush input buffer on io->fd : %i", io->fd);
	return 0; /* TODO: check errno */
    }

    return 1;
}



int LIO_Serial_SetAM(LIO_Serial * io, LIO_Serial_Transfermode * trm)
{
    /* check the io/trm anchor */
    if ((io == NULL) || (trm == NULL)) {
	DBG_ERROR("missing pointer to either io structure or transfermode");
	return 0;
    }

    switch (*trm) {

      case LIO_SERIAL_TRM_BYTEWISE:
      case LIO_SERIAL_TRM_WORDWISE:
      case LIO_SERIAL_TRM_DWORDWISE:
      case LIO_SERIAL_TRM_DDWORDWISE:
      case LIO_SERIAL_TRM_FAST:
	io->trm  = *trm;
	return 1;
	break;		/* paranoia */

      default:		/* unknown access mode */
	return 0;

    }
}



int LIO_Serial_GetAM(LIO_Serial * io, LIO_Serial_Transfermode * trm)
{
    /* check the io/trm anchor */
    if ((io == NULL) || (trm == NULL)) {
	DBG_ERROR("missing pointer to either io structure or transfermode");
	return 0;
    }

    *trm = io->trm;
    return 1;
}



int LIO_Serial_GetPort(LIO_Serial * io, int * port)
{
    /* check the io/port anchor */
    if ((io == NULL) || (port == NULL)) {
	DBG_ERROR("missing pointer to either io structure or port");
	return 0;
    }

    *port = io->port;
    return 1;
}



int LIO_Serial_Read(LIO_Serial * io, int bwt, size_t size, unsigned char * data)
{
    unsigned char	c;
    int			count;

    /* check the io/data anchor */
    if ((io == NULL) || (data == NULL)) {
	DBG_ERROR("missing pointer to either io structure or data");
	return 0;
    }

    for (count = 0; count < size; count ++) {

	if (LIOS_CanRead(io->fd, 0, bwt)) {

	    if (read(io->fd, &c, 1) != 1) {

		return 0;

	    }

	    data[count] = c;

	} else {

	    /* tcflush(io->fd, TCIFLUSH); */
	    return 0;

	}

    }

    return 1;
}



int LIO_Serial_Write(LIO_Serial * io, int cwt, size_t size, const unsigned char * data)
{
    int	count;

    /* check the io/data anchor */
    if ((io == NULL) || (data == NULL)) {
	DBG_ERROR("missing pointer to either io structure or data");
	return 0;
    }

    if (io->trm == LIO_SERIAL_TRM_FAST) {

	if (write(io->fd, data, size) != size) {
	    return 0;
	}

	return 1;
    }

    /* discard input data from previous commands */
    tcflush(io->fd, TCIFLUSH);

    for (count = 0; count < size; count ++) {

	if (LIOS_CanWrite(io->fd, cwt, 1000)) {

	    if (write(io->fd, data + count, 1) != 1) {

		return 0;

	    }

	} else {

	    /* tcflush(io->fd, TCIFLUSH); */
	    return 0;

	}

    }

    return 1;
}



int LIO_Serial_Close(LIO_Serial * io)
{
    /* check the io anchor */
    if (io == NULL) {
	DBG_ERROR("missing pointer to io structure");
	return 0;
    }

    if (close(io->fd) < 0) {
	DBG_ERROR("can't close file on io->fd : %i", io->fd);
	return 0; /* TODO: check errno */
    }

    LIOS_Clear(io);

    return 1;
}



void LIO_Serial_Delete(LIO_Serial * io)
{
    free(io); /* TODO: check if io->fd is in use */
}





static int LIOS_CanRead(int fd, unsigned delay_ms, unsigned timeout_ms)
{
    fd_set		rfds;	/* all observed file descriptores for reading */
    struct timeval	tv;	/* timeout interval */

    if (delay_ms > 0) {
	/*
	* TODO: we have to replace usleep() through more portable code; see
	*       "Anwendungen entwickeln unter Linux"; M.K. Johnson; site 193
	*/
	usleep((unsigned long)(delay_ms * 1000));
    }

    tv.tv_sec = timeout_ms / 1000;
    tv.tv_usec = (timeout_ms % 1000) * 1000;

    FD_ZERO(&rfds);
    FD_SET(fd, &rfds);

    if (select(fd + 1, &rfds, NULL, NULL, &tv) <= 0) {
	return 0;
    }

    if (FD_ISSET(fd, &rfds)) {
	return 1;
    } else {
	return 0;
    }
}



static int LIOS_CanWrite(int fd, unsigned delay_ms, unsigned timeout_ms)
{
    fd_set		wfds;	/* all observed file descriptores for write */
    struct timeval	tv;	/* timeout interval */

    if (delay_ms > 0) {
	/*
	* TODO: we have to replace usleep() through more portable code; see
	*       "Anwendungen entwickeln unter Linux"; M.K. Johnson; site 193
	*/
	usleep((unsigned long)(delay_ms * 1000));
    }

    tv.tv_sec = timeout_ms / 1000;
    tv.tv_usec = (timeout_ms % 1000) * 1000;

    FD_ZERO(&wfds);
    FD_SET(fd, &wfds);

    if (select(fd + 1, NULL, &wfds,  NULL, &tv) <= 0) {
	return 0;
    }

    if (FD_ISSET(fd, &wfds)) {
	return 1;
    } else {
	return 0;
    }
}



static void LIOS_DeviceName(int port, unsigned char * filename, int length)
{
#ifdef  OS_LINUX
    snprintf(filename, length, "/dev/ttyS%d", port - 1);
#endif
#ifdef  OS_SOLARIS
    snprintf(filename, length, "/dev/cua/%c", 'a' + port - 1);
#endif
#ifdef  OS_CYGWIN32
    /* TODO: snprintf(filename, length, "COM%d", port); */
    sprintf(filename, "COM%d", port);
#endif
}



static void LIOS_Clear(LIO_Serial * io)
{
    io->fd    = -1;
    io->port  = 0;
}
