/*
    Copyright (C) 2003 by Stephan Linz <linz@li-pro.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Bosten, MA 02111-1307 USA
*/

/* $Id: $ */

#include <avr/io.h>
#include <avrhal/delay.h>
#include <avrhal/ow.h>


#define	__OW_PULSE(TB, TA)													\
		{																	\
			/* drive low level (output port) */								\
			sbi(AVRHAL_LIB_OW_PORT_DDR, AVRHAL_LIB_OW_DQBIT);				\
			/* for (TB) us */												\
			delay_us((TB));													\
			/* release bus (input port) */									\
			cbi(AVRHAL_LIB_OW_PORT_DDR, AVRHAL_LIB_OW_DQBIT);				\
			/* and wait for (TA) us */										\
			delay_us((TA));													\
		}


extern unsigned char	_ow_devices_active;


/*
 * Generate a 1-Wire reset, return 0 if no presence detect pulse was found,
 * return 1 otherwise.
 * (NOTE: Does not handle alarm presence)
 */
unsigned char _ow_reset(void)
{
	/* be shure DQ is Tri-state (external
	 * pull-up resistore drives high level) */
	cbi(AVRHAL_LIB_OW_PORT, AVRHAL_LIB_OW_DQBIT);

	if (bit_is_clear(AVRHAL_LIB_OW_PORT_IN, AVRHAL_LIB_OW_DQBIT))
		return 0;					/* detect low level on bus error, so
									   we tell there is no device */

	__OW_PULSE(480, 60);			/* 1-Wire reset pulse */

	if (bit_is_clear(AVRHAL_LIB_OW_PORT_IN, AVRHAL_LIB_OW_DQBIT)) {
		delay_us(420);				/* detect low level on device present */
		return 1;					/* wait to fill out time slot */
	} else {
		delay_us(420);				/* sorry, there is no device */
		return 0;					/* wait to fill out time slot */
	}
}

/*
 * Send a 1-Wire write bit. Provide 10us recovery time.
 */
void _ow_bit_write(unsigned char bit)
{
	/* be shure DQ is Tri-state (external
	 * pull-up resistore drives high level) */
	cbi(AVRHAL_LIB_OW_PORT, AVRHAL_LIB_OW_DQBIT);

	if (bit) {
		__OW_PULSE(6, 64);			/* 1-Wire write `1` pulse */
	} else {
		__OW_PULSE(60, 10);			/* 1-Wire write `0` pulse */
	}
}

/*
 * Read a bit from the 1-Wire bus and return it. Provide 10us recovery time.
 */
unsigned char _ow_bit_read(void)
{
	/* be shure DQ is Tri-state (external
	 * pull-up resistore drives high level) */
	cbi(AVRHAL_LIB_OW_PORT, AVRHAL_LIB_OW_DQBIT);

	__OW_PULSE(6, 9);				/* 1-Wire read pulse */

	if (bit_is_clear(AVRHAL_LIB_OW_PORT_IN, AVRHAL_LIB_OW_DQBIT)) {
		delay_us(55);				/* detect low level on `0` bit */
		return 0;					/* wait to fill out time slot */
	} else {
		delay_us(55);				/* detect high level on `1` bit */
		return 1;					/* wait to fill out time slot */
	}
}

/*
 * Send even one byte over the 1-Wire bus.
 */
unsigned char ow_byte_write(unsigned char byte)
{
	int i;
	for (i = 0; i < 8; i++) _ow_bit_write(byte & BV(i));
	return 1;
}

/*
 * Read even one byte from the 1-Wire bus and return it.
 */
unsigned char ow_byte_read(void)
{
	int i;
	unsigned char rb = 0;
	for (i = 0; i < 8; i++) rb |= _ow_bit_read() ? BV(i) : 0;
	return rb;
}


/*
 * generic bus (re-)initialization, returns boolean value of success
 */
unsigned char ow_init(void)
{
	/* input port, Tri-state (default mode) */
	cbi(AVRHAL_LIB_OW_PORT,		AVRHAL_LIB_OW_DQBIT);
	cbi(AVRHAL_LIB_OW_PORT_DDR,	AVRHAL_LIB_OW_DQBIT);

	_ow_devices_active = 0;			/* reset device list */

	if (!_ow_reset())
		return 0;					/* it seems, there is no device anymore */

	return 1;
}


/*************************************EndOfFile***************************
$Id: $
vim:tw=78:ts=4:sw=4:ai:
*/
