/*
    Copyright (C) 2003 by Stephan Linz <linz@li-pro.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Bosten, MA 02111-1307 USA
*/
/*
 * The most stuf here is taken from Dalles Semiconductor
 * Application Note 187 "1-Wire Search Algorithm".
 *
 * Remarks:    20021027 Stephan Linz <linz@li-pro.net>
 *             adapted to AVR-Ctrl board designed by
 *             http://www.mikrocontroller.com
 */

/* $Id: $ */

#include <avrhal/crc.h>
#include <avrhal/ow.h>

extern unsigned char	_ow_reset(void);
extern void				_ow_bit_write(unsigned char bit);
extern unsigned char	_ow_bit_read(void);

ow_device_st *_ow_devices = NULL;
unsigned char _ow_devices_active = 0;

/*
 * global search state
 */
#if AVRHAL_LIB_OW_DEVS == OW_MANY_DEV
static unsigned char	LastDiscrepancy,
						LastFamilyDiscrepancy,
						LastDeviceFlag;
#endif


/*
 * Perform the 1-Wire search algorithm on the 1-Wire bus
 * using the existing search state.
 */
#if AVRHAL_LIB_OW_DEVS == OW_MANY_DEV
static unsigned char _ow_rom_search(unsigned char *ROM_NO)
{
	unsigned char	id_bit_number;
	unsigned char	last_zero, rom_byte_number, search_result;
	unsigned char	id_bit, cmp_id_bit;
	unsigned char	rom_byte_mask, search_direction;

	// initialize for search
	id_bit_number	= 1;
	last_zero		= 0;
	rom_byte_number	= 0;
	rom_byte_mask	= 1;
	search_result	= 0;

	// if the last call was not the last one
	if (!LastDeviceFlag) {

		if (!_ow_reset()) {

			// there is no device, reset the search state
			LastDiscrepancy =
				LastDeviceFlag =
				LastFamilyDiscrepancy = 0;

			return 0;
		}

		ow_byte_write(0xf0);		// SEARCH ROM command

		// loop to do search
		do {

			// read a bit and its complement
			id_bit		= _ow_bit_read();
			cmp_id_bit	= _ow_bit_read();

			// check for no devices on 1-Wire
			if ((id_bit == 1) && (cmp_id_bit == 1))
				break;
			else {

				// all devices coupled have 0 or 1
				if (id_bit != cmp_id_bit)
					search_direction = id_bit;	// bit write value for search
				else {

					// if this discrepancy if before the Last Discrepancy
					// on a previous next then pick the same as last time
					if (id_bit_number < LastDiscrepancy)
						search_direction = ((ROM_NO[rom_byte_number] & rom_byte_mask) > 0);
					else
						// if equal to last pick 1, if not then pick 0
						search_direction = (id_bit_number == LastDiscrepancy);

					// if 0 was picked then record its position in LastZero
					if (search_direction == 0) {

						last_zero = id_bit_number;

						// check for Last Discrepancy in family
						if (last_zero < 9)
							LastFamilyDiscrepancy = last_zero;
					}
				}

				// set or clear the bit in the ROM byte rom_byte_number
				// with mask rom_byte_mask
				if (search_direction == 1)
					ROM_NO[rom_byte_number] |= rom_byte_mask;
				else
					ROM_NO[rom_byte_number] &= ~rom_byte_mask;

				// serial number search direction write bit
				_ow_bit_write(search_direction);

				// increment the byte counter id_bit_number
				// and shift the mask rom_byte_mask
				id_bit_number++;
				rom_byte_mask <<= 1;

				// if the mask is 0 then go to new SerialNum byte
				// rom_byte_number and reset mask
				if (rom_byte_mask == 0) {
					/******************* crc8_ow(); *******************/
					rom_byte_number++;
					rom_byte_mask = 1;
				}
			}
		} while (rom_byte_number < sizeof(ow_rom_code_st));	// loop until
															// through all
															// ROM bytes 0..7

		// if the search was successful then
		if (!((id_bit_number < 65) ||
					(crc8_ow(ROM_NO, sizeof(ow_rom_code_st)) != 0))) {

			// search successful so set LastDiscrepancy,
			// LastDeviceFlag, and search_result
			LastDiscrepancy = last_zero;

			// check for last device
			if (LastDiscrepancy == 0)
				LastDeviceFlag = 1;

			search_result = 1;
		}
	}

	// if no device found then reset counters so next 'search' will be
	// like a first
	if (!search_result || !ROM_NO[0]) {
		LastDiscrepancy =
			LastDeviceFlag =
			LastFamilyDiscrepancy =
			search_result = 0;
	}

	return search_result;
}
#endif

/*
 * Find the 'first' device on the 1-Wire bus.
 */
#if AVRHAL_LIB_OW_DEVS == OW_MANY_DEV
static unsigned char _ow_rom_search_first(unsigned char *ROM_NO)
{
	// reset the search state
	LastDiscrepancy =
		LastDeviceFlag =
		LastFamilyDiscrepancy = 0;

	return _ow_rom_search(ROM_NO);
}
#endif

/*
 * Find the 'next' device on the 1-Wire bus.
 */
#if AVRHAL_LIB_OW_DEVS == OW_MANY_DEV
static unsigned char _ow_rom_search_next(unsigned char *ROM_NO)
{
	// leave the search state alone
	return _ow_rom_search(ROM_NO);
}
#endif


unsigned char ow_rom_search(ow_device_st *ow_dev)
{
#if AVRHAL_LIB_OW_DEVS == OW_MANY_DEV
	unsigned char rslt;
#endif

	_ow_devices			= ow_dev;
	_ow_devices_active	= 0;

#if AVRHAL_LIB_OW_DEVS == OW_ONE_DEV
	if (!_ow_reset())			// it seems, there is no device anymore
		return 0;

	ow_byte_write(0x33);		// READ ROM command
	if (ow_buf_read((unsigned char *)(&(_ow_devices[_ow_devices_active].rom_code)),
				sizeof(ow_rom_code_st), 1))
		_ow_devices_active	= 1;
#elif AVRHAL_LIB_OW_DEVS == OW_MANY_DEV
	rslt = _ow_rom_search_first(
			(unsigned char *)(&(_ow_devices[_ow_devices_active].rom_code)));

	while (rslt) {
		rslt = _ow_rom_search_next(
				(unsigned char *)(&(_ow_devices[++_ow_devices_active].rom_code)));
	}
#else
#error *** missing or incorrect AVRHAL_LIB_OW_DEVS setting
#endif

	return _ow_devices_active;
}

/*************************************EndOfFile***************************
$Id: $
vim:tw=78:ts=4:sw=4:ai:
*/
