/*
    Copyright (C) 2003 by Stephan Linz <linz@li-pro.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Bosten, MA 02111-1307 USA
*/

/* $Id: $ */

/*
   avrhal/ds1820.h - DS1820 support
 */

#ifndef _AVRHALL_DS1820_H_
#define _AVRHALL_DS1820_H_ 1

#if defined(HAVE_CONFIG_H) && defined(AVRHAL_LIB_SRC)
#include <config.h>
#else
#include <avrhal/config.h>
#endif

#ifndef __ASSEMBLER__

/** \defgroup avrhal_ow_ds1820 Temperatursensor DS1820/DS1822
    \ingroup avrhal_ow
    \anchor a_avrhal_ow_ds1820
    \code #include <avrhal/ds1820.h> \endcode

    Diese Headerdatei deklariert einen einfachen Low-Level Zugang zu den
    Temperatursensoren DS1820/DS1822 von Dallas Semiconductors ber
    den One Wire Bus. Um die hier aufgefhrten Funktionen benutzen zu knnen,
    mu der One Wire Bus initialisiert und betriebsbereit sein. Es ist keine
    weitere Initialisierung dieses Teils der Bibliothek notwendig.

    \see avrhal_ow

    \todo Integration der \em ALARM Ereignisbehandlung. Anlenung an
    CodeVision's unsigned char ds1820_set_alarm(unsigned char *addr,
    signed char temp_low, signed char temp_high). */

/** \name Konvertierung und Abfrage */

/*@{*/

/** \def DS1820_FAMILY
    \ingroup avrhal_ow_ds1820
    \brief One Wire Familien Code von DS1820/DS1822 */
#define DS1820_FAMILY		0x10

#ifdef __cplusplus
extern "C" {
#endif

/** \fn unsigned char ds1820_convert(unsigned char device)
    \ingroup avrhal_ow_ds1820
    \brief Konvertierung der Temperatur auslsen.

    Mit dieser Funktion wird der ber \em device adressierte Temperatursensor
    zu einer neuen Temperaturkonvertierung aufgefordert. Die Funktion wartet
    mit Hilfe von ow_ready() auf das Ende dieser Konvertierung.

    Die Gerteauswahl \em device antspricht dem Index in dem von
    ow_rom_search() gefllten Feld mit erkannten Gerteadressen.

    \param device Gerteindex der internen Gerteverwaltung.

    \return Die Funktion ds1820_convert() gibt eine Null(0) zurck, wenn ein
    Fehler aufgetreten ist. Ansosnsten wird ein Wert ungleich Null(!=0) fr
    eine erfolgreiche Konvertierung zurckgegeben. */
extern unsigned char ds1820_convert(unsigned char device);

/** \fn int ds1820_temp10_C(unsigned char device)
    \ingroup avrhal_ow_ds1820
    \brief Temperatur auslesen.

    Mit dieser Funktion wird der ber \em device adressierte Temperatursensor
    mit Hilfe von ds1820_convert() zu einer neuen Temperaturkonvertierung
    aufgefordert und das Ergebnis davon eingelesen. Die Temperatur wird in
    eine dezimale Grad-Celsius Zahl berfhrt und zurckgegeben.

    Die Gerteauswahl \em device antspricht dem Index in dem von
    ow_rom_search() gefllten Feld mit erkannten Gerteadressen.

    \param device Gerteindex der internen Gerteverwaltung.

    \return Die Funktion ds1820_temp10_C() gibt die aktuelle Temperatur in
    Grad-Celsius oder -9999 bei Auftreten eines Fehlers zurck. */
extern signed int ds1820_temp10_C(unsigned char device);

#ifdef __cplusplus
}
#endif

/*@}*/

/** \name CodeVision Konformitt */

/*@{*/

#ifdef __cplusplus
extern "C" {
#endif

/** \fn int ds1820_temperature_10(unsigned char *addr)
    \ingroup avrhal_ow_ds1820

    \codevision

    Aliasfunktion fr ds1820_temp10_C() wobei bei dieser Funktion direkt
    ber die 8 Byte lange One Wire Gerteadresse auf den Temperatursensor
    zugegriffen wird. Die Gerteadresse wird dabei automatisch in die
    Gertenummer berfhrt.
    
    \param addr One Wire Gerteadresse.
    
    \return Die Funktion ds1820_temperature_10() besitzt den Rckgabewert
    von ds1820_temp10_C(). */
extern int ds1820_temperature_10(unsigned char *addr);

#ifdef __cplusplus
}
#endif

/*@}*/

#endif /* __ASSEMBLER__ */

#endif /* _AVRHALL_DS1820_H_ */
