/*
    Copyright (C) 2003 by Stephan Linz <linz@li-pro.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Bosten, MA 02111-1307 USA
*/

/* $Id: $ */

/*
   avrhal/key.h - basic KEY pad i/o
 */

#ifndef _AVRHALL_KEY_H_
#define _AVRHALL_KEY_H_ 1

#if defined(HAVE_CONFIG_H) && defined(AVRHAL_LIB_SRC)
#include <config.h>
#else
#include <avrhal/config.h>
#endif

#ifndef __ASSEMBLER__

#include <avr/io.h>

/** \defgroup avrhal_key Tastaturfeld
    \anchor a_avrhal_key
    \code #include <avrhal/key.h> \endcode

    Diese Headerdatei deklariert einen einfachen Low-Level Zugang zu exakt
    5 Tasten an einem konfigurierbaren Port. Es werden ausschlielich nur die
    5 Bits beachtet, an denen auch Taster angeschlossen sind (Bit 3..7). Die
    restlichen Bit 0..2 des KEY Port bleiben unberhrt.

    Vor der Benutzung der KEY Funktionen, mu dieser Teil der Bibliothek mit
    key_init() initialisiert werden. Die Bibliothek stellt fr die Ermittlung
    gedrckter Tasten die Funktion key_scancode() bereit. Jeder Taste ist im
    Scancode ein Bit zugeordnet. Fr die einfache Auswertung sind diese Codes
    Bestandteil dieser Deklaration.

    \todo berfhrung der Initialisierung nach Sektion \c .init1
    \todo Parametrisierung ber mehrere Ports verstreuter Bits.
    \todo Untersttzung fr Tastaturmatrix.
    \todo Wenn machbar, dann eine Ereigniskontrolle (call-back)
          einfhren. */

/** \name Scancodes */

/*@{*/

/** \def KEY_SCAN_T1
    \ingroup avrhal_key
    \brief Bitwert fr Taste T1, erste Taste von links. */
#define KEY_SCAN_T1		_BV(7)

/** \def KEY_SCAN_T2
    \ingroup avrhal_key
    \brief Bitwert fr Taste T2, zweite Taste von links. */
#define KEY_SCAN_T2		_BV(6)

/** \def KEY_SCAN_T3
    \ingroup avrhal_key
    \brief Bitwert fr Taste T3, dritte Taste von links. */
#define KEY_SCAN_T3		_BV(5)

/** \def KEY_SCAN_T4
    \ingroup avrhal_key
    \brief Bitwert fr Taste T4, vierte Taste von links. */
#define KEY_SCAN_T4		_BV(4)

/** \def KEY_SCAN_T5
    \ingroup avrhal_key
    \brief Bitwert fr Taste T5, fnfte Taste von links. */
#define KEY_SCAN_T5		_BV(3)

/** \def KEY_SCAN_ALL
    \ingroup avrhal_key
    \brief Bitwert fr alle Tasten T1 bis T5 (Tastenmaske). */
#define KEY_SCAN_ALL							\
(	KEY_SCAN_T1							\
|	KEY_SCAN_T2							\
|	KEY_SCAN_T3							\
|	KEY_SCAN_T4							\
|	KEY_SCAN_T5	)

/*@}*/

/** \name Low-Level Zugriff */

/*@{*/

#ifdef __cplusplus
extern "C" {
#endif

/** \fn void key_init(void)
    \ingroup avrhal_key
    \brief (Re-)Initialisierung des Tastaturfeldes.

    Initialisiert den KEY Port zur Biteingabe (alle 5 Bits). Dabei bleiben
    die unteren Bits unbehandelt.
    
    \return Die Funktion key_init() besitzt keinen Rckgabewert. */
static inline void key_init(void)
{
	AVRHAL_LIB_KEY_PORT_DDR &= ~KEY_SCAN_ALL;
}

/** \fn unsigned char key_scancode(void)
    \ingroup avrhal_key
    \brief Tastatur Sancode einlesen.

    Liet den aktuellen Zustand der Tasten ein und liefert einen
    entsprechenden Scancode. Die folgende Abbildung zeigt die Tasten T1 bis
    T5 und die zugehrigen Bitwerte als Teil des Scancodes, wenn T2, T3 und
    T5 gedrckt sind.

    \image html key_pad.png "Tastenfeld mit Scancode"
    \image latex key_pad.eps "Tastenfeld mit Scancode" width=7.5cm

    Es wird immer ein Byte als Scancode zurckgegeben. Dieses setzt sich wie
    folgt zusammen:

    \verbatim
    AVR-Ctrl key numbers:
    _______________________________
   |                               |
   |      LCD ...                  |
   |_______________________________|
     _____ _____ _____ _____ _____
    |  T1 |  T2 |  T3 |  T4 |  T5 |
    |_____|_____|_____|_____|_____|
    |\___/|\___/|\___/|\___/|\___/|
    ||___|||___|||___|||___|||___||
    |/___\|/___\|/___\|/___\|/___\|

     ___ ___ ___ ___ ___ ___ ___ ___
    |   |   |   |   |   |   |   |   |
    | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 | - scancode (Byte)
    |___|___|___|___|___|___|___|___|
      |   |   |   |   |   |_______|
      |   |   |   |   |       |________ unused
      |   |   |   |   |
      |   |   |   |   |________________ KEY_SCAN_T5
      |   |   |   |
      |   |   |   |____________________ KEY_SCAN_T4
      |   |   |
      |   |   |________________________ KEY_SCAN_T3
      |   |
      |   |____________________________ KEY_SCAN_T2
      |
      |________________________________ KEY_SCAN_T1
    \endverbatim

    \return Die Funktion key_scancode() gibt den aktuellen Scancode zurck. */
static inline unsigned char key_scancode(void)
{
	return (AVRHAL_LIB_KEY_PORT_IN & KEY_SCAN_ALL);
}

#ifdef __cplusplus
}
#endif

/*@}*/

#endif /* __ASSEMBLER__ */

#endif /* _AVRHALL_KEY_H_ */
