/*
    Copyright (C) 2003 by Stephan Linz <linz@li-pro.net>

    Some code comes from Brajer Vlado <vlado.brajer@kks.s-net.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Bosten, MA 02111-1307 USA
*/

/* $Id: $ */

/*
   avrhal/lcd.h - basic LCD ascii i/o
 */

#ifndef _AVRHALL_LCD_H_
#define _AVRHALL_LCD_H_ 1

#if defined(HAVE_CONFIG_H) && defined(AVRHAL_LIB_SRC)
#include <config.h>
#else
#include <avrhal/config.h>
#endif

#ifndef __ASSEMBLER__

/** \defgroup avrhal_lcd Alphanumerisches LCD
    \anchor a_avrhal_lcd
    \code #include <avrhal/led.h> \endcode

    Diese Headerdatei deklariert zunchst einen Low-Level Zugang fr ein
    alphanumerisches LCD. Es werden dabei 1 bis 4 Zeilen Displays mit bis zu
    64 Zeichen je Zeile untersttzt. Diese LCD besitzen in aller Regel einen
    eigenen Controller, der auf dem Typ HD44780 von Hitachi basiert. Hier
    eine Liste von Vergleichstypen. Displays mit einem dieser Chips sollten
    mit dem hier bereitgestellten Code zusammenarbeiten:
	- Hitachi \b HD44780 / \b HD44780S / \b HD44780U / \b LM018L
	- Samsung \b KS0066U
	- Seiko-Epson \b SED1278
    
    Es wird immer von einer Nibble Kommunikation (4 Bit) mit dem LCD
    ausgegangen. Der LCD Port ist auf \c \avrhal_lib_lcd_port festgelegt. Die
    Steuersignale RS, RD und EN sind am selben LCD Port zum
    bersetzungszeitpunkt frei definierbar. Das eine unbenutzte Bit wird zur
    Laufzeit auch als solches bercksichtigt. Damit ergibt sich eine
    konfigurierte Bitverteilung von:
	- \c \avrhal_lib_lcd_port Bit 7..4: Daten Nibble (DB7..4)
	- \c \avrhal_lib_lcd_port Bit \avrhal_lib_lcd_unused_bit: unbenutzt
	- \c \avrhal_lib_lcd_port Bit \avrhal_lib_lcd_en: Signal EN
	- \c \avrhal_lib_lcd_port Bit \avrhal_lib_lcd_rd: Signal RD
	- \c \avrhal_lib_lcd_port Bit \avrhal_lib_lcd_rs: Signal RS

    Vor der Benutzung der LCD Funktionen, mu dieser Teil der Bibliothek mit
    lcd_init() initialisiert werden. Hierbei wird der Typ des LCD Moduls
    verschlsselt bergeben. Fr die weitere Benutzung des LCD existieren
    abstrahierte aktionsbezogene Funktionen zum Positionieren des Cursors und
    Ausgeben von Zahlen und Zeichenketten. Nach Datenbuch des Controllers
    werden folgende Funktionen des LCD untersttzt (LCD instruction set):
	- \b Clear \b Display
	- \b Return \b Home
	- \b Display \b ON/OFF \b Control
	- \b Set \b CGRAM \b Address
	- \b Set \b DDRAM \b Address
	- \b Read \b Busy \b Flag \b & \b Address (wenn gewnscht)

    Der letzte Punkt wird nur untersttzt, wenn zum bersetzungszeitpunkt
    das Polling aktiviert wurde. Bei deaktiviertem Polling wird mit einem
    Timeout Verfahren gearbeitet. Dieses sollte mit allen LCD Typen
    funktionieren. Es kann aber vorkommen, da manche Controller den gegebenen
    Zeitrahmen nicht einhalten. Das Timeout Verfahren sollte dann zum Einsatz
    kommen, wenn man kleineren Code bentigt und weniger auf Ablaufzeit
    achtet.

    Um zwischen Polling und Timeout Verfahren zu entscheiden, mu man ... .
    Das \b \if LCD_POLLING Polling \elseif LCD_DELAY Timeout \elseif ???
    \endif \endif \endif Verfahren wurde aktiviert.

    Fr die Erstellung dieses Codes wurden folgende Referenzen benutzt:
	- http://bray.velenje.cx/avr/lcd/lcd.html
	- http://www.doc.ic.ac.uk/~ih/doc/lcd/index.html
	- http://www.repairfaq.org/filipg/LINK/F_LCD_HD44780.html
	- http://home.iae.nl/users/pouweha/lcd/lcd.shtml

    \todo berfhrung von Teilen der Initialisierung nach Sektion \c .init1
    \todo Untersttzung eigener frei definierter Grafikzeichen.
    \todo Balken- und Punktgrafik hnlich den LED Funktionen.
    \todo Wenn mglich mehr Hardwarebeschleunigung. */

/** \name Low-Level Zugriff / Brajer Vlado Konformitt */

/*@{*/

#ifdef __cplusplus
extern "C" {
#endif

/** \fn void lcd_init(unsigned char lcd_matrix)
    \ingroup avrhal_lcd
    \brief (Re-)Initialisirung des LCD Moduls.

    Mit Aufruf dieser Funktion wird das am LCD Port \c \avrhal_lib_lcd_port
    angeschlossene LCD Modul neu (re-)initialisiert. Dabei wird die 4 Bit
    Datenbertrag zum Modul fest eingestellt. Der Cursor wird inkremental
    eingestellt.

    Neben den Einstellungen des Ports selbst und der Konfiguration des LCD
    Moduls werden wichtige interne Parameter gesetzt, um die Koordination
    der Datenandressen in Abhngigkeit des angeschlossenen LCD Moduls zu
    realisieren. Hierzu wird der bergebene Typenschlssel des LCD Moduls
    \em lcd_matrix benutzt. Die Bedeutung der Bits im Typenschlssel zeigt
    die folgende Abbildung.

    \verbatim
     ___ ___ ___ ___ ___ ___ ___ ___
    |   |   |   |   |   |   |   |   |
    | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 | - lcd_matrix
    |___|___|___|___|___|___|___|___|
      |___|   |___________________|
        |               |______________ chars per line (CPL):
        |                                 0x00 : n/d [1]
        |                                    . . .
        |                                 0x10 : 16 characters
        |                                    . . .
        |                                 0x14 : 20 characters
        |                                    . . .
        |                                 0x3f : 63 characters [1]
        |
        |______________________________ number of lines (L):
                                            00 : like 11 [2]
                                            01 : 1 line
    [1] see ToDo list                       10 : 2 lines
    [2] compatibility to CodeVision         11 : 4 lines
    \endverbatim

    Der Typenschlssel wird intern dazu benutzt, die Basisadresse (Anfang)
    einer jeden Zeile im Daten RAM des LCD zu berechnen. Somit kann eine X/Y
    Positionierung in die reale Adresse umgerechnet werden. Auch andere
    positionsabhngige Operationen sind erst damit mglich. Die folgende
    Abbildung zeigt den Vergleich von drei verschiedenen 16 Zeichen LCD mit
    1, 2 und 4 Zeilen.

    \verbatim
    line __      __ base address in display data ram (DDRAM)
           |    |
           |   _|__  ____  ____  ____  ____  __ . . . __  ____
           V  | V  ||    ||    ||    ||    ||           ||    | - L=01
              |    ||    ||    ||    ||    ||           ||    |
           1  |0x00||0x01||0x02||0x03||0x04||           ||0x0F|
              |____||____||____||____||____||__ . . . __||____|
               ____  ____  ____  ____  ____  __ . . . __  ____
              |    ||    ||    ||    ||    ||           ||    | - L=10
              |    ||    ||    ||    ||    ||           ||    |
           1  |0x00||0x01||0x02||0x03||0x04||           ||0x0F|
              |____||____||____||____||____||__ . . . __||____|
              |    ||    ||    ||    ||    ||           ||    |
              |    ||    ||    ||    ||    ||           ||    |
           2  |0x40||0x41||0x42||0x43||0x44||           ||0x4F|
              |____||____||____||____||____||__ . . . __||____|
               ____  ____  ____  ____  ____  __ . . . __  ____
              |    ||    ||    ||    ||    ||           ||    | - L=11
              |    ||    ||    ||    ||    ||           ||    |
           1  |0x00||0x01||0x02||0x03||0x04||           ||0x0F|
              |____||____||____||____||____||__ . . . __||____|
              |    ||    ||    ||    ||    ||           ||    |
              |    ||    ||    ||    ||    ||           ||    |
           2  |0x40||0x41||0x42||0x43||0x44||           ||0x4F|
              |____||____||____||____||____||__ . . . __||____|
              |    ||    ||    ||    ||    ||           ||    |
              |    ||    ||    ||    ||    ||           ||    |
           3  |0x10||0x11||0x12||0x13||0x14||           ||0x1F|
              |____||____||____||____||____||__ . . . __||____|
              |    ||    ||    ||    ||    ||           ||    |
              |    ||    ||    ||    ||    ||           ||    |
           4  |0x50||0x51||0x52||0x53||0x54||           ||0x5F|
              |____||____||____||____||____||__ . . . __||____|
    \endverbatim

    \codevision
    Da bei CodeVision zwei Headerdateien fr verschiedene LCD Typen
    existieren, ist die hier deklarierte API nicht voll kompatibel. Bei einer
    Codeberfhrung von CodeVision zu dieser Bibliothek mu lcd_init()
    gegebenenfalls angepat werden. Vorrangig wird immer ein 4 Zeilen LCD
    angenommen.

    \param lcd_matrix definiert den Typ des LCD Moduls. Siehe oben.

    \return Die Funktion lcd_init() besitzt keinen Rckgabewert.

    \todo Beseitigung der unklaren Nummerierung fr die Anzahl Zeichen je
    Zeile in \em lcd_matrix, oder anders: Was ist bei Anzahl gleich Null(0) ?
    \todo Differenzierung in real existierende LCD Typen und somit wegfall
    der Berechnung von Anfangsadressen im Datenspeicher. Die Adressen werden
    zum bersetzungszeitpunkt ermittelt und fest einprogrammiert. */
extern void lcd_init(unsigned char lcd_matrix);

/** \fn void lcd_cls(void)
    \ingroup avrhal_lcd
    \brief LCD Anzeige lschen.

    Mit dieser Funktion wird der aktuell dargestellte Inhalt des
    Datenspeichers im LCD gelscht (mit Leerzeichen beschrieben).
    
    \lcdinstset Es kommt die LCD Funktion \b Clear \b Display zur Anwendung:
    \verbatim
     RS   RD   DB7  DB6  DB5  DB4  DB3  DB2  DB1  DB0
    [ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 1 ]
    \endverbatim
 
    \return Die Funktion lcd_cls() besitzt keinen Rckgabewert. */
extern void lcd_cls(void);

/** \fn void lcd_home(void)
    \ingroup avrhal_lcd
    \brief Cursor auf erste beschreibbare Position zurck.

    Mit dieser Funktion wird der Cursor von der aktuellen Position auf die
    aller erste beschreibbare zurckgesetzt. Diese Position befindet sich in
    der linken oberen Ecke im Anzeigebereich des LCD und entspricht der
    Adresse \c 0x00 im Datenspeicher des LCD. Alle weieren Zeichenausgaben am
    LCD erfolgen ab dieser neuen Position.
    
    \lcdinstset Es kommt die LCD Funktion \b Return \b Home zur Anwendung:
    \verbatim
     RS   RD   DB7  DB6  DB5  DB4  DB3  DB2  DB1  DB0
    [ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 1 ][ 0 ]
    \endverbatim

    \return Die Funktion lcd_home() besitzt keinen Rckgabewert. */
extern void lcd_home(void);

/** \fn void lcd_goto(unsigned char addr)
    \ingroup avrhal_lcd
    \brief Cursor auf eine beliebige Position setzen.

    Mit dieser Funktion wird der Cursor von der aktuellen Position auf eine
    mit \em addr bergebene neue gesetzt. Diese Position entspricht exakt
    einer Adresse im Datenspeicher des LCD und sollte den Wert \c 0x7f nicht
    berschreiten. Alle weieren Zeichenausgaben am LCD erfolgen ab dieser
    neuen Position.
    
    \lcdinstset Es kommt die LCD Funktion \b Set \b DDRAM \b Address zur
    Anwendung:
    \verbatim
     RS   RD   DB7  DB6  DB5  DB4  DB3  DB2  DB1  DB0
    [ 0 ][ 0 ][ 1 ][AC6][AC5][AC4][AC3][AC2][AC1][AC0]

        AC[6..0] equ. (addr & 0x7f)
    \endverbatim
 
    \param addr gibt die neue Adresse im Datenspeicher des LCD an.
    
    \return Die Funktion lcd_goto() besitzt keinen Rckgabewert. */
extern void lcd_goto(unsigned char addr);

/** \fn void lcd_ctrl(unsigned char lcd_control)
    \ingroup avrhal_lcd
    \brief Kontrolle der LCD Arbeitsweise.

    Mit dieser Funktion wird die Arbeitsweise des LCD verndert. Hierzu
    wird das bergebene Kontrollbyte \em lcd_control wie folgt interpretiert.

    \verbatim
     ___ ___ ___ ___ ___ ___ ___ ___
    |   |   |   |   |   |   |   |   |
    | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 | - lcd_control
    |___|___|___|___|___|___|___|___|
      |_______________|   |   |   |____ cursor blink (CB):
              |           |   |           0 : off
              |           |   |           1 : on
              |           |   |
              |           |   |________ cursor on/off (COO):
              |           |               0 : off
              |           |               1 : on
              |           |
              |           |____________ display on/off (DOO):
              |                           0 : off
              |                           1 : on
              |
              |________________________ unused
    \endverbatim

    \lcdinstset Es kommt die LCD Funktion \b Display \b ON/OFF \b Control zur
    Anwendung:
    \verbatim
     RS   RD   DB7  DB6  DB5  DB4  DB3  DB2  DB1  DB0
    [ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 0 ][ 1 ][ D ][ C ][ B ]

        D equ. DOO
	C equ. COO
	B equ. CB
    \endverbatim

    \param lcd_control ist das Kontrollbyte mit der neuen LCD Arbeitsweise.

    \return Die Funktion lcd_ctrl() besitzt keinen Rckgabewert.
    
    \todo Es fehlen noch die Kontrolle ber die Bewegungsrichtung des Cursors
    und das Verschieben des Datenspeichers im LCD. Dabei soll die LCD Funktion
    \b Entry \b Mode \b Set verwendet werden. */
extern void lcd_ctrl(unsigned char lcd_control);

/** \fn void lcd_cleol(void)
    \ingroup avrhal_lcd
    \brief LCD Zeile ab Cursor lschen.

    Mit dieser Funktion wird der Inhalt der aktuellen Zeile ab aktueller
    Cursorposition bis zum Ende gelscht (mit Leerzeichen beschrieben).
    
    \return Die Funktion lcd_cleol() besitzt keinen Rckgabewert. */
extern void lcd_cleol(void);

/** \fn void lcd_putch(unsigned char data)
    \ingroup avrhal_lcd
    \brief Zeichenausgabe ab Cursor.

    Mir dieser Funktion wird das mit \em data bergebene ASCII Zeichen
    auf dem LCD direkt und ohne Filterung an der aktuellen Anzeigeposition
    am LCD ausgegeben, also in den Daten RAM des LCD geschrieben. Dabei
    wird die interne Cursorsteuerung entsprechend korrigiert. Der Cursor
    wird um ein Zeichen nach rechts verschoben. Am Zeilenende erfolgt der
    automatische Umbruch. Am Displayende erfolgt der automatische Sprung
    zur ersten beschreibbaren Position wie unter Verwendung von
    \em lcd_home().

    \param data ist das auszugebende Zeichen.
    
    \return Die Funktion lcd_putch() besitzt keinen Rckgabewert. */
extern void lcd_putch(unsigned char data);

/** \fn void lcd_print10(unsigned long x)
    \ingroup avrhal_lcd
    \brief Ausgabe einer 10-stelligen Dezimalzahl.

    Mit dieser Funktion wird die mit \em x bergebene 32 Bit positive Ganzzahl
    unter Zuhilfenahme von lcd_putch() ab der aktuellen Anzeigeposition am
    LCD mit exakt 10 Stellen als Dezimalzahl auf dem LCD ausgegeben. Fhrende
    Nullen werden immer mit ausgegeben! Es erfolgt \b keine Kontrolle des
    Wertebereiches!

    \param x ist die auszugebende Zahl.
    
    \return Die Funktion lcd_print10() besitzt keinen Rckgabewert. */
extern void lcd_print10(unsigned long x);

/** \fn void lcd_print2(unsigned char x)
    \ingroup avrhal_lcd
    \brief Ausgabe einer 2-stelligen Dezimalzahl.

    Mit dieser Funktion wird die mit \em x bergebene 8 Bit positive Ganzzahl
    unter Zuhilfenahme von lcd_putch() ab der aktuellen Anzeigeposition am
    LCD mit exakt 2 Stellen als Dezimalzahl auf dem LCD ausgegeben. Fhrende
    Nullen werden immer mit ausgegeben! Es erfolgt eine Kontrolle des
    Wertebereiches, wobei sich \em x zwischen 0 und 99 bewegen darf.

    \param x ist die auszugebende Zahl.
    
    \return Die Funktion lcd_print2() besitzt keinen Rckgabewert. */
extern void lcd_print2(unsigned char x);

/** \fn void lcd_print3(unsigned int x)
    \ingroup avrhal_lcd
    \brief Ausgabe einer 3-stelligen Dezimalzahl.

    Mit dieser Funktion wird die mit \em x bergebene 16 Bit positive Ganzzahl
    unter Zuhilfenahme von lcd_putch() ab der aktuellen Anzeigeposition am
    LCD mit exakt 3 Stellen als Dezimalzahl auf dem LCD ausgegeben. Fhrende
    Nullen werden immer mit ausgegeben! Es erfolgt eine Kontrolle des
    Wertebereiches, wobei sich \em x zwischen 0 und 999 bewegen darf.

    \param x ist die auszugebende Zahl.
    
    \return Die Funktion lcd_print3() besitzt keinen Rckgabewert. */
extern void lcd_print3(unsigned int x);

/** \fn void lcd_print5(unsigned int x)
    \ingroup avrhal_lcd
    \brief Ausgabe einer 5-stelligen Dezimalzahl.

    Mit dieser Funktion wird die mit \em x bergebene 16 Bit positive Ganzzahl
    unter Zuhilfenahme von lcd_putch() ab der aktuellen Anzeigeposition am
    LCD mit exakt 5 Stellen als Dezimalzahl auf dem LCD ausgegeben. Fhrende
    Nullen werden immer mit ausgegeben! Es erfolgt \b keine Kontrolle des
    Wertebereiches!

    \param x ist die auszugebende Zahl.
    
    \return Die Funktion lcd_print5() besitzt keinen Rckgabewert. */
extern void lcd_print5(unsigned int x);

/** \fn void lcd_printbin(unsigned char x)
    \ingroup avrhal_lcd
    \brief Ausgabe einer 8-stelligen Binrzahl.

    Mit dieser Funktion wird die mit \em x bergebene 8 Bit positive Ganzzahl
    unter Zuhilfenahme von lcd_putch() ab der aktuellen Anzeigeposition am
    LCD mit exakt 8 Stellen als Binrzahl auf dem LCD ausgegeben. Dabei wird
    mit dem MSB begonnen.

    \param x ist die auszugebende Zahl.
    
    \return Die Funktion lcd_printbin() besitzt keinen Rckgabewert. */
extern void lcd_printbin(unsigned char x);

/** \fn void lcd_printf(const char * format, ...)
    \ingroup avrhal_lcd
    \brief Primitiver Nachbau der Std-Lib-C printf() Funktion.

    Mit dieser sehr einfach gehaltenen Nachbildung der aus der Standard C
    Bibliothek bekannten \em printf() Funktion kann man ASCII Zeichenketten
    mit einfachen Formatanweisungen auf dem LCD ab der aktuellen
    Anzeigeposition am LCD ausgeben. Hierbei wird ausschlielich nur die
    Funktion \em lcd_putch() benutzt. Bekannt sind folgende Formatierer ohne
    Formaterweiterungen wie fhrende Nullen oder Stellenbegrenzer:
	- \c c     : gibt ein ASCII Zeichen aus
	- \c u und
	  \c d     : gibt eine Zahl vom Typ \c int als 5-stellige Dezimalzahl
	             mit fhrenden Nullen aus
	- \c x     : gibt eine Zahl vom Typ \c int als 2-stellige
	             Hexadezimalzahl mit fhrenden Nullen aus

    \warning Diese Funktion wird in spteren Versionen der Bibliothek
    eventuell durch den generischen \em printf() Mechanismus der Standard C
    Bibliothek fr AVR Controller, \em avrlibc, stillschweigend ersetzt.

    \param format ist der Formatstring.
    
    \return Die Funktion lcd_printf() besitzt keinen Rckgabewert. */
extern void lcd_printf(const char * format, ...);

/** \fn void lcd_printhex(unsigned char x)
    \ingroup avrhal_lcd
    \brief Ausgabe einer 2-stelligen Hexadezimalzahl.

    Mit dieser Funktion wird die mit \em x bergebene 8 Bit positive Ganzzahl
    unter Zuhilfenahme von lcd_putch() ab der aktuellen Anzeigeposition am
    LCD mit exakt 2 Stellen als Hexadezimalzahl auf dem LCD ausgegeben.
    Fhrende Nullen werden immer mit ausgegeben!

    \param x ist die auszugebende Zahl.
    
    \return Die Funktion lcd_printhex() besitzt keinen Rckgabewert. */
extern void lcd_printhex(unsigned char x);

/** \fn void lcd_putstr(unsigned char * data)
    \ingroup avrhal_lcd
    \brief Zeichenkettenausgabe ab Cursor.

    Mir dieser Funktion wird die ber \em data referenzierte ASCII
    Zeichenkette unter Zuhilfenahme von lcd_putch() auf dem LCD direkt
    und ohne Filterung an der aktuellen Anzeigeposition am LCD ausgegeben.

    \param data zeigt auf die auszugebende Zeichenkette.
    
    \return Die Funktion lcd_putstr() besitzt keinen Rckgabewert. */
extern void lcd_putstr(unsigned char * data);

/** \fn void lcd_control(unsigned char d, unsigned char c, unsigned char b)
    \ingroup avrhal_lcd
    \brief Vereinfachter Aufruf von lcd_ctrl().

    \warning Diese Funktion existiert aus Grnden der Abwrtskompatibilitt
    und wird in einer der nchsten Verffentlichungen der Bibliothek
    stillschweigend entfallen.

    \param d gibt mit Eins(1) an, ob das Display aktiv ist.
    \param c gibt mit Eins(1) an, ob der Cursor aktiv ist.
    \param b gibt mit Eins(1) an, ob der Cursor blinkt.

    \return Die Funktion lcd_control() besitzt keinen Rckgabewert. */
static inline void lcd_control(unsigned char d, unsigned char c, unsigned char b)
{
	lcd_ctrl((unsigned char)(	( (d & 1) << 2 )
				|	( (c & 1) << 1 )
				|	  (b & 1)		));
}

#ifdef __cplusplus
}
#endif

/*@}*/

/** \name CodeVision Konformitt */

/*@{*/

#ifdef __cplusplus
extern "C" {
#endif

/** \fn void lcd_gotoxy(unsigned char x, unsigned char y)
    \ingroup avrhal_lcd
    \brief Cursor auf eine beliebeige X-Y-Position setzen.

    \codevision

    Mit dieser Funktion wird der Cursor von der aktuellen Position auf eine
    mit \em x und \em y neu definierte gesetzt. Diese Position setzt sich
    aus einer X-Koordinate und einer Y-Koordinate zusammen, wobei X das
    Zeichen in einer Zeile und Y die Zeile selbst ist. Die Zhlweise beginnt
    bei beiden Koordinaten mit 0 ab der linken oberen Ecke. Es gilg also
    immer: lcd_gotoxy(0,0); entspricht lcd_goto(0); entspricht lcd_home();

    Diese Funktion rechnet den kartesischen Koordinatenwert in Abhngigkeit
    des mit lcd_init() konfigurierten LCD Types in eine absolute Adresse im
    Datenspeicher des LCD um und benutzt dann die Funktion lcd_goto(), um den
    Cursor neu zu positionieren. Bei dieser Berechnung erfolgt kein Test auf
    gltigen Wertebereich!

    \param x gibt die neue X-Koordinate im Anzeigebereich des LCD an.
    \param y gibt die neue Y-Koordinate im Anzeigebereich des LCD an.
    
    \return Die Funktion lcd_gotoxy() besitzt keinen Rckgabewert. */
extern void lcd_gotoxy(unsigned char x, unsigned char y);

/** \fn void lcd_write_byte(unsigned char addr, unsigned char data)
    \ingroup avrhal_lcd
    \brief Datenbyte ausgeben.

    \codevision

    Schreibt das Byte \em data an die Adresse \em addr im LCD Zeichengenerator
    oder Datenspeicher.

    \par Beispiel:
    \include example_lcd_write_byte.c

    \param addr gibt die Adresse im Datenspeicher des LCD an.
    \param data ist das auszugebende Zeichen.
    
    \return Die Funktion lcd_write_byte() besitzt keinen Rckgabewert. */
extern void lcd_write_byte(unsigned char addr, unsigned char data);

/** \fn void _lcd_write_data(unsigned char data)
    \ingroup avrhal_lcd
    \brief Kommandobyte ausgeben.

    \codevision

    Schreibt das Byte \em data in das LCD Anweisungsregister (instruction
    register). Diese Funktion kann zum Modifizieren der LCD Konfiguration
    benutzt werden.

    \par Beispiel:
    \include example__lcd_write_data.c

    \param data ist das auszugebende Zeichen.
    
    \return Die Funktion _lcd_write_data() besitzt keinen Rckgabewert. */
extern void _lcd_write_CMD(unsigned char);
static inline void _lcd_write_data(unsigned char data)
{
	_lcd_write_CMD(data);
}

/** \fn void lcd_clear(void)
    \ingroup avrhal_lcd
    \brief LCD Anzeige lschen.

    \codevision

    Aliasfunktion fr lcd_cls(). Lscht die Anzeige des LCD und setzt den
    Cursor auf die linke obere Ecke.
    
    \return Die Funktion lcd_clear() besitzt keinen Rckgabewert. */
static inline void lcd_clear(void)
{
	lcd_cls();
}

/** \fn void lcd_putchar(char c)
    \ingroup avrhal_lcd
    \brief Zeichenausgabe ab Cursor.

    \codevision

    Aliasfunktion fr lcd_putch(). Zeigt das Zeichen \em c an der aktuellen
    Anzeigeposition am LCD an.

    \param c ist das auszugebende Zeichen.
    
    \return Die Funktion lcd_putchar() besitzt keinen Rckgabewert. */
static inline void lcd_putchar(char c)
{
	lcd_putch((unsigned char)c);
}

/** \fn void lcd_puts(char *str)
    \ingroup avrhal_lcd
    \brief Zeichenkettenausgabe ab Cursor.

    \codevision

    Aliasfunktion fr lcd_putstr(). Zeigt die Zeichenkette, auf die \em str
    zeigt, ab der aktuellen Anzeigeposition am LCD an.
    
    \param str zeigt auf die auszugebende Zeichenkette.
    
    \return Die Funktion lcd_puts() besitzt keinen Rckgabewert. */
static inline void lcd_puts(char *str)
{
	lcd_putstr((unsigned char *)str);
}

/** \fn void lcd_putsf(char flash *str)
    \ingroup avrhal_lcd
    \brief Zeichenkettenausgabe ab Cursor (aus dem Flash).

    \codevision

    Zeigt die Zeichenkette, auf die \em str im Flash zeigt, ab der aktuellen
    Anzeigeposition am LCD an. In dieser LCD Implementierung ist diese
    Funktionalitt nicht realisiert.
    
    \param str zeigt auf die auszugebende Zeichenkette im Flash.
    
    \return Die Funktion lcd_putsf() besitzt keinen Rckgabewert. */
#define flash
static inline void lcd_putsf(char flash *str __attribute__ ((unused)))
{
	/* NOP */
}
#undef flash

/** \fn unsigned char lcd_read_byte(unsigned char addr)
    \ingroup avrhal_lcd
    \brief Datenbyte einlesen.

    \codevision

    Liest ein Byte aus dem LCD Zeichengenerator oder Daten RAM. In dieser LCD
    Implementierung ist diese Funktionalitt nicht realisiert.
    
    \param addr ist die zu verwendende Adresse im LCD.
    
    \return Die Funktion lcd_read_byte() gibt immer Null(0) zurck. */
static inline unsigned char lcd_read_byte(
		unsigned char addr __attribute__ ((unused)))
{
	return 0;
}

/** \fn void _lcd_ready(void)
    \ingroup avrhal_lcd
    \brief LCD Bereitschaftstest.

    \codevision

    Wartet, bis das LCD zur Kommunikation bereit ist. Diese Funktion mu immer
    vor dem Schreiben von Daten an das LCD mit _lcd_write_data() aufgerufen
    werden.
    
    \return Die Funktion _lcd_ready() besitzt keinen Rckgabewert. */
#if AVRHAL_LIB_LCD_COM == LCD_POLLING
extern void _lcd_check_BF(void);
static inline void _lcd_ready(void)
{
	_lcd_check_BF();
}
#elif AVRHAL_LIB_LCD_COM == LCD_DELAY
#warning *** LCD_POLLING not defined -- use simulated _lcd_ready()
extern void delay_us(unsigned int);
static inline void _lcd_ready(void)
{
	delay_us(2000);
}
#else
#error *** missing or incorrect AVRHAL_LIB_LCD_COM setting
#endif

#ifdef __cplusplus
}
#endif

/*@}*/

#endif /* __ASSEMBLER__ */

#endif /* _AVRHALL_LCD_H_ */
