/*
    Copyright (C) 2003 by Stephan Linz <linz@li-pro.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Bosten, MA 02111-1307 USA
*/

/* $Id: $ */

/*
   avrhal/led.h - basic LED bar i/o
 */

#ifndef _AVRHALL_LED_H_
#define _AVRHALL_LED_H_ 1

#if defined(HAVE_CONFIG_H) && defined(AVRHAL_LIB_SRC)
#include <config.h>
#else
#include <avrhal/config.h>
#endif

#ifndef __ASSEMBLER__

/** \defgroup avrhal_led LED Balken
    \anchor a_avrhal_led
    \code #include <avrhal/led.h> \endcode

    Diese Headerdatei deklariert zunchst einen Low-Level Zugang fr einen LED
    Balken. Dabei werden immer exakt 8 Bits (LEDs) des LED Ports benutzt. Der
    LED Port ist auf \c \avrhal_lib_led_port festgelegt.

    Darber hinaus existiert eine zur Laufzeit parametrisierbare LED Grafik
    mit Normierung. Hierbei kann man den Nullpunkt zwischen der untersten und
    der obersten LED frei definieren und wenn ntig nachtrglich verschieben.
    Somit ist die Anzeige von positiven und negativen Werten mglich. Diese
    knnen dann je nach Einstellung als Punkt (eine LED) oder als Balken in
    Bezug zum Nullpunkt dargestellt werden. Mehr dazu bei der Beschreibung der
    Funktion led_graph_control().

    Die Arbeitsweise der LED Grafik wird durch zwei interne Parameter
    bestimmt, welche mit led_graph_control() verndert werden knnen. Alle
    Ausgaben mssen dann mit led_graph_put() erfolgen.

    Vor der Benutzung der LED Funktionen, mu dieser Teil der Bibliothek mit
    led_init() initialisiert werden.

    \todo berfhrung der Initialisierung nach Sektion \c .init1
    \todo Parametrisierung ber mehrere Ports verstreuter Bits. */

/** \name Low-Level Zugriff */

/*@{*/

#ifdef __cplusplus
extern "C" {
#endif

/** \fn void led_init(void)
    \ingroup avrhal_led
    \brief LED Port initialisieren.

    Initialisiert den LED Port zur Byteausgabe (alle 8 Bits).

    \return Die Funktion led_init() besitzt keinen Rckgabewert. */
static inline void led_init(void)
{
	AVRHAL_LIB_LED_PORT_DDR = 0xff;
}

/** \fn void led_cls(void)
    \ingroup avrhal_led
    \brief LED Balken lschen.

    Setzt alle Bits auf den inaktiven Zustand (OFF(\avrhal_lib_led_off)
    value).
    
    \return Die Funktion led_cls() besitzt keinen Rckgabewert. */
static inline void led_cls(void)
{
	AVRHAL_LIB_LED_PORT = AVRHAL_LIB_LED_OFF;
}

/** \fn void led_put(unsigned char bitset)
    \ingroup avrhal_led
    \brief LED Balken beschreiben.

    Setzt alle Bits auf den angegebenen Wert bitset.
    
    \param bitset ist vom Typ \c unsigned \c char und wird als auszugebendes
                  Bitmuster interpretiert.

    \return Die Funktion led_put() besitzt keinen Rckgabewert. */
static inline void led_put(unsigned char bitset)
{
	AVRHAL_LIB_LED_PORT = bitset;
}

/** \fn unsigned char led_get(void)
    \ingroup avrhal_led
    \brief LED Balken zurcklesen.

    Gibt den Wert der aktuell gesetzten LED Ausgabebits zurck.
    
    \return Mit der Funktion led_get() wird das aktuelle Bitmuster vom LED
    Port zurckgelesen. */
static inline unsigned char led_get(void)
{
	return (AVRHAL_LIB_LED_PORT_IN);
}

/** \fn void led_clear(void)
    \ingroup avrhal_led
    \brief LED Balken lschen.

    Macht das Gleiche wie led_cls(). Setzt alle Bits auf den inaktiven
    Zustand (OFF(\avrhal_lib_led_off) value).
    
    \return Die Funktion led_clear() besitzt keinen Rckgabewert. */
static inline void led_clear(void)
{
	led_cls();
}

#ifdef __cplusplus
}
#endif

/*@}*/

/** \name Balkengrafik */

/*@{*/

#ifdef __cplusplus
extern "C" {
#endif

/** \fn void led_graph_control(int led_maxlevel, signed char led_options)
    \ingroup avrhal_led
    \brief Verndert die interne Arbeitsweise der LED Grafik.

    Als Basisinitialisierung der LED Grafik ist die unterste LED als Nullpunkt
    eingestellt (Bit 0) und es gibt keine Normierung. Die Funkltion
    led_graph_put() verhlt sich somit wie led_put().

    Mit der Funktion led_graph_control() kann diese Arbeitsweise verndert
    werden. Hierzu wird der maximal darstellbarer Realwert \em led_maxlevel
    fr die Normierung und das Optionsfeld \em led_options bergeben. Durch
    das Optionsfeld wird zwischen Punkt- oder Balkengrafik entschieden (GM)
    und es wird das Bit fr den Bezugs- bzw. Nullpunkt der Grafik festgelegt
    (ZPB). Die Bedeutung der Bits im Optionsfeld zeigt die folgende Abbildung.

    \verbatim
     ___ ___ ___ ___ ___ ___ ___ ___
    |   |   |   |   |   |   |   |   |
    | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 | - led_options
    |___|___|___|___|___|___|___|___|
      |___________|   |_______|   |____ graph mode (GM):
              |           |                 0 : dot graph
              |           |                 1 : bar graph
              |           |    
              |           |____________ zero point bit (ZPB):
              |                           000 : LED Port Bit 0
              |                           001 : LED Port Bit 1
              |                           010 : LED Port Bit 2
              |                             . . .
              |                           111 : LED Port Bit 7
              |
              |________________________ unused
    \endverbatim

    Die folgende Abbildung zeigt ein kurzes Beispiel. Dabei wird die
    Balkengrafik (GM=1) der zugehrigen Punktgrafik (GM=0) gegenber
    gestellt. Der Nullpunkt ist mit ZBP=010 auf das LED Port Bit 2 festgelegt.
    Der mit der Funktion led_graph_put() bergebene Anzeigewert
    \c bitset_level wird mit dem Wert \c led_maxlevel normiert an LED Port
    Bit 6 ausgegeben. Die LED Port Bits 0 bis 1 sind der negative
    Anzeigebereich.

    \image html led_graph.png "LED Grafik Beispiel"
    \image latex led_graph.eps "LED Grafik Beispiel" width=7.5cm

    \param led_maxlevel ist der maximal darstellbarer Realwert und wird fr
    die Normierung benutzt. Der Wert mu ungleich Null und grer 8 oder
    kleiner -8 sein. Erfllt der Wert diese Regel nicht, wird die Normierung
    deaktiviert. Der Wert kann auch negativ sein. Dann sollte der Bezugspunkt
    (Nullpunkt) auf einen Wert grer Null gesetzt werden.

    \param led_options definiert die Arbeitsweise der LED Grafik. Siehe oben.

    \return Die Funktion led_graph_control() besitzt keinen Rckgabewert. */
extern void led_graph_control(int led_maxlevel, signed char led_options);

/** \fn void led_graph_put(int bitset_level)
    \ingroup avrhal_led
    \brief Gibt einen Wert als LED Grafik aus.

    Je nach Parametrisierung durch led_graph_control() wird mit dieser
    Funktion der bergebene Wert zunchst normiert und dann als Grafik
    ausgegeben.
    
    \note Ist keine Normierung eingestellt, so wird der angegebene Wert mit
    led_put() direkt als Bitmuster ausgegeben!

    \param bitset_level ist eine positive oder negative ganze Zahl, die je
    nach Konfiguration der LED Garfik angezeigt wird.

    \return Die Funktion led_graph_put() besitzt keinen Rckgabewert. */
extern void led_graph_put(int bitset_level);

#ifdef __cplusplus
}
#endif

/** \def LED_BAR
    \ingroup avrhal_led
    \brief Bytewert zum Aktivieren der LED Balkengrafik. */
#define	LED_BAR			(_BV(0))

/** \def LED_POINT
    \ingroup avrhal_led
    \brief Bytewert zum Aktivieren der LED Punktgrafik. */
#define	LED_POINT		(~(_BV(0)))

/** \def LED_MARGIN(M)
    \ingroup avrhal_led
    \brief Bytewert zum Festlegen des Null- bzw. Bezubgspunktes fr die LED Grafik.

    \param M ist dabei das (LED-)Bit fr den Nullpunkt. */
#define LED_MARGIN(M)		(((M) & 0x7) << 1)

/*@}*/

#endif /* __ASSEMBLER__ */

#endif /* _AVRHALL_LED_H_ */
