/*
    Copyright (C) 2003 by Stephan Linz <linz@li-pro.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Bosten, MA 02111-1307 USA
*/

/* $Id: $ */

/*
   avrhal/ow.h - basic One Wire bus i/o
 */

#ifndef _AVRHALL_OW_H_
#define _AVRHALL_OW_H_ 1

#if defined(HAVE_CONFIG_H) && defined(AVRHAL_LIB_SRC)
#include <config.h>
#else
#include <avrhal/config.h>
#endif

#ifndef __ASSEMBLER__

#define __need_NULL
#define __need_size_t
#include <stddef.h>

/** \defgroup avrhal_ow One Wire Buszugriff
    \anchor a_avrhal_ow
    \code #include <avrhal/ow.h> \endcode

    Diese Headerdatei deklariert einen einfachen Low-Level Zugang zum
    Eindrahtkommunikationssystem \em One \em Wire von Dallas. Mit dessen
    Hilfe knnen daran angeschlossene Sensoren, Speicher oder sonstige
    I/O-Bausteine angesprochen werden. Der One Wire Bus ist an
    \c \avrhal_lib_ow_port Bit \avrhal_lib_ow_dqbit angeschlossen. Zur
    Zeit sind folgende One Wire Gerte ber ein eigenes API ansprechbar:
	- \ref avrhal_ow_ds1820

    Dieser Teil der Bibliothek mu zunchst mit ow_init() initialisiert
    werden, um in der internen Gerteverwaltung alle bekannten Gerte zu
    vergessen. Der Initialisierungsdurchlauf kann also mehrfach erfolgen.
    Nach einer (Re-)Initialiserung mssen mit ow_rom_search() alle am
    One Wire Bus befindlichen Gerte gesucht und deren \em ROM \em Code in
    eine Liste hinterlegt werden. Hierzu mu der Benutzer der Bibliothek
    ausreichend Speicherplatz als Feld vom Typ \c ow_device_st
    bereitstellen, da alle Funktionen dieser One Wire Implementierung
    keinerlei Tests auf Speicherberlufe durchfren. Das Feld mu also fr
    jedes physisch am Bus befindliche One Wire Gert ein Feldeintrag
    vorhalten, in dem dann, nach erfolgreichem Suchen, der zugehrige ROM
    Code abgelegt ist.

    Fr die Erstellung dieses Codes wurden folgende Referenzen benutzt:
    \ref supp_ow_footnote_one	"[1]",
    \ref supp_ow_footnote_two	"[2]",
    \ref supp_ow_footnote_three	"[3]",

    \anchor supp_ow_footnote_one
    \note <b>[1]</b> Dallas Semiconductor Application Note 159 "Ultra-Reliable 1-Wire Communications"
    \anchor supp_ow_footnote_two
    \note <b>[2]</b> Dallas Semiconductor Application Note 187 "1-Wire Search Algorithm"
    \anchor supp_ow_footnote_three
    \note <b>[3]</b> Dallas Semiconductor Application Note 126 "1-Wire Communications Through Software"

    \todo berfhrung von Teilen der Initialisierung nach Sektion \c .init1
    \todo Integration einer One Wire \em ALARM Ereignisbehandlung.
    \todo Code-Optimierung bei Eingertebenutzung (Wegfall der Umfangreichen
    ROM Code Suche, Wrapper Macros). */

/** \name Low-Level Zugriff und Gerteverwaltung */

/*@{*/

/** \struct ow_rom_code_s ow.h "avrhal/ow.h"
    \ingroup avrhal_ow
    \brief One Wire ROM Code

    Dieser strukturierte Typ stellt ausreichend Felder fr einen One Wire ROM
    Code bereit. Dieser ROM Code ist nach \ref supp_ow_footnote_two "[2]"
    64 Bit breit und besteht aus 1 Byte fr die Gerteklasse, 6 Byte fr eine
    Seriennummer und 1 Byte CRC Summe ber die vorangegangenen 7 Byte. */
struct ow_rom_code_s {
	unsigned char		family;		/**< Gerteklasse */
	unsigned char		serial[6];	/**< Seriennummer */
	unsigned char		crc;		/**< 8 Bit CRC Summe */
};

/** \typedef ow_rom_code_st
    \ingroup avrhal_ow
    \brief Typvereinbarung fr einen One Wire ROM Code

    \see ow_rom_code_s */
typedef struct ow_rom_code_s	ow_rom_code_st;

/** \struct ow_dev_flags_s ow.h "avrhal/ow.h"
    \ingroup avrhal_ow
    \brief One Wire Gerteflags */
struct ow_dev_flags_s {
	unsigned	alarm:1;	/**< Alarm */
	unsigned	parpower:1;	/**< parasitre Energieversorgung */
	unsigned	unused:6;
};

/** \struct ow_device_s ow.h "avrhal/ow.h"
    \ingroup avrhal_ow
    \brief One Wire Gerteeintrag

    Jedes One Wire Gert besitzt neben dem allgemeingltigen ROM Code noch
    weitere funktionsbezogene Status- und Kontrollinformationen. Diese werden
    in diesem strukturierten Typ zusammengefat. Die Wahl zwischen der
    Benutzung des Statusbytes oder der differenzierten Flags hngt von der
    jeweiligen Gerteklasse ab. Mehr dazu findet man im Handbuch des
    zum ROM Code zugehrigen One Wire Gertes. */
struct ow_device_s {
	ow_rom_code_st		rom_code;	/**< One Wire ROM Code */
	union {
	  unsigned char		status;		/**< Statusbyte (z.B. DS2405) */
	  struct ow_dev_flags_s	flags;		/**< Gerteflags */
	};
};

/** \typedef ow_device_st
    \ingroup avrhal_ow
    \brief Typvereinbarung fr einen One Wire Gerteeintrag

    \see ow_device_s */
typedef struct ow_device_s	ow_device_st;

#ifdef __cplusplus
extern "C" {
#endif

/** \fn unsigned char ow_init(void)
    \ingroup avrhal_ow
    \brief (Re-)Initialisirung des One Wire Moduls.

    Mit Aufruf dieser Funktion wird der One Wire Bus durch ein entsprechendes
    Busreset initialisiert und die interne Gerteverwaltung zurckgesetzt.
    
    \return Die Funktion ow_init() gibt eine Null(0) zurck, wenn keines der
    angeschlossenen One Wire Gerte das Busreset beantwortet. Ansonsten wird
    eine Eins(1) zurckgegeben. */
extern unsigned char ow_init(void);

/** \fn unsigned char ow_rom_search(ow_device_st *ow_dev)
    \ingroup avrhal_ow
    \brief One Wire ROM Code Ermittlung

    Mit dieser Funktion wird der One Wire Bus so wie in
    \ref supp_ow_footnote_two "[2]" beschrieben nach Gerten durchsucht,
    wobei deren ROM Codes ermittelt und in einem Feld vom Typ \c ow_device_st
    hinterlegt werden. Der fr dieses Feld notwendige Speicher wird \b nicht
    von der Bibliothek bereitgestellt, sondern mu vom Benutzer ber den
    Zeiger \em ow_dev referenziert werden.
    
    \warning Fr die interne Gerteverwaltung wird das ber \em ow_dev
    referenziert Feld durch die Bibliothek weiter benutzt; auch ber die
    Arbeit von ow_rom_search() hinaus. Sollte sich also die Gre dieses
    Feldes, welche immer in direktem Bezug zu der Anzahl physisch vorhandener
    One Wire Gerte steht, verndern, so mu eine Reinitialisierung mittels
    ow_init() und ow_rom_search() erfolgen.
    
    \param ow_dev Zeiger auf ein Feld fr die interne Gerteverwaltung.
    
    \return Die Funktion ow_rom_search() gibt eine Null(0) zurck, wenn kein
    One Wire Gert gefunden wurde. Ansonsten wird die Anzahl der gefundenen
    One Wire Gerte zurckgegeben. Diese \b mu der Gre des durch \em ow_dev
    referenzierten Feldes entsprechen. */
extern unsigned char ow_rom_search(ow_device_st *ow_dev);

/** \fn unsigned char ow_byte_read(void)
    \ingroup avrhal_ow
    \brief Lesen eines Bytes vom One Wire Bus

    Mit dieser Funktion wird die Signalfolge zum Lesen von genau einem Byte
    vom One Wire Bus ausgelst. Das jeweilige One Wire Gert, von dem gelesen
    werden soll, mu zuvor durch andere Funktionen entsprechend vorbereitet
    werden (z.B. mit ow_function()).
    
    \return Die Funktion ow_byte_read() gibt das vom Bus gelesene Byte
    zurck. */
extern unsigned char ow_byte_read(void);

/** \fn unsigned char ow_byte_write(unsigned char byte)
    \ingroup avrhal_ow
    \brief Schreiben eines Bytes zum One Wire Bus

    Mit dieser Funktion wird die Signalfolge zum Schreiben von genau einem
    Byte am One Wire Bus ausgelst. Das zu schreibende Byte wird mit \em byte
    bergeben. Das jeweilige One Wire Gert, in das geschrieben werden soll,
    mu zuvor durch andere Funktionen entsprechend vorbereitet werden (z.B.
    mit ow_function()).
    
    \param byte zu schreibendes Byte.
    
    \return Die Funktion ow_byte_write() gibt immer eine Eins(1) zurck, da
    der Erfolg oder Mierfolg einer Bytebertragung am One Wire Bus nicht
    direkt kontrolliert werden kann. */
extern unsigned char ow_byte_write(unsigned char byte);

#ifdef __cplusplus
}
#endif

/*@}*/

/** \name Spezialfunktionen */

/*@{*/

#ifdef __cplusplus
extern "C" {
#endif

/** \fn unsigned char ow_ready(void)
    \ingroup avrhal_ow
    \brief Primitiver Bereitschaftstest

    Mit dieser Funktion kann die Bereitschaft eines One Wire Gertes nach
    Ausfhrung eines vorher abgesetzten Kommandos durch einen einfachen
    Bitlesezyklus getestet werden. Viele One Wire Gerte zeigen so die
    Beendigung einer Aktion an, so z.B. die einmalige Temperaturmessung
    im Sensor DS1820.

    Zum Schutz gegen endloses Warten aufgrund fehlerhafter One Wire Gerte
    oder Strungen beim Lesen eines Bits vom Bus, ist der Testzyklus mit
    einer Zeitschleife von 1 Sekunde versehen. Nach erfolglosem Test ber
    diese Zeit hinaus wird automatisch abgebrochen!

    \return Die Funktion ow_ready() gibt eine Null(0) zurck, wenn vom One
    Wire Gert innerhalb der Zeitschleife von 1 Sekunde keine Bereitschaft
    angezeigt wurde. Ansonsten wird ein Wert ungleich Null(!=0) fr einen
    erfolgreichen Bereitschaftstest zurckgegeben. */
extern unsigned char ow_ready(void);

/** \fn unsigned char ow_buf_read(unsigned char *buf, size_t buf_size, unsigned char crc_included)
    \ingroup avrhal_ow
    \brief Bytes vom One Wire Bus in einen Puffer lesen

    Mit dieser Funktion knnen unter Zuhilfenahme von ow_byte_read() so viele
    Bytes hintereinander vom One Wire Bus in den ber \em buf referenzierten
    Puffer gelesen werden, wie ber \em buf_size angegeben sind. Auf Wunsch
    erfolgt zustzlich mit crc8_ow() ein Test der im letzten Byte enthaltenen
    CRC Summe, wenn \em crc_included ungleich Null ist. Das jeweilige One Wire
    Gert, von dem gelesen werden soll, mu zuvor durch andere Funktionen
    entsprechend vorbereitet werden (z.B. mit ow_function()).

    \param buf Zeiger auf den Anfang des Puffers.
    \param buf_size Anzahl der Bytes im Puffer (Gre).
    \param crc_included Anzeige ber enthaltene CRC Summe.

    \return Die Funktion ow_buf_read() gibt bei fehlerhafter bertragung eine
    Null(0) Zurck. Ansonsten wird eine Eins(1) zurckgegeben. */
extern unsigned char ow_buf_read(unsigned char *buf, size_t buf_size,
				 unsigned char crc_included);

/** \fn unsigned char ow_buf_write(unsigned char *buf, size_t buf_size, unsigned char crc_included)
    \ingroup avrhal_ow
    \brief Bytes aus einem Puffer zum One Wire Bus schreiben

    Mit dieser Funktion knnen unter Zuhilfenahme von ow_byte_write() so viele
    Bytes hintereinander aus den ber \em buf referenzierten Puffer zum One
    Wire Bus geschrieben werden, wie ber \em buf_size angegeben sind. Auf
    Wunsch erfolgt zustzlich mit crc8_ow() die Berechnung einer CRC Summe im
    letzten Byte des Puffers, wenn \em crc_included ungleich Null ist. Das
    jeweilige One Wire Gert, in das geschrieben werden soll, mu zuvor durch
    andere Funktionen entsprechend vorbereitet werden (z.B. mit ow_function()).

    \warning Soll die Berechnung der CRC Summe innerhalb dieser Funktion
    erfolgen, dann darf das letzte Byte im Puffer keine relevanten Nutzdaten
    enthalten. Das letzte Byte \c buf[(buf_size-1)] wird mit der CRC Summe
    ber die vorgelagerten Bytes \c buf[0..(buf_size-2)] berschrieben!

    \param buf Zeiger auf den Anfang des Puffers.
    \param buf_size Anzahl der Bytes im Puffer (Gre).
    \param crc_included Anzeige ber enthaltene CRC Summe.

    \return Die Funktion ow_buf_read() gibt bei fehlerhafter bertragung eine
    Null(0) zurck. Ansonsten wird eine Eins(1) zurckgegeben. */
extern unsigned char ow_buf_write(unsigned char *buf, size_t buf_size,
				  unsigned char crc_included);

/** \fn unsigned char ow_function(unsigned char code, unsigned char device)
    \ingroup avrhal_ow
    \brief Bytes aus einem Puffer zum One Wire Bus schreiben

    Mit dieser Funktion wird unter Zuhilfenahme von ow_byte_write() und
    ow_buf_write() ein neues Gertekommando \em code an das ber \em device
    referenzierte One Wire Gert initiert. Hierbei stellt \em device einen
    Index in dem der Funktion ow_rom_search() bergebenen Feld vom Typ
    \c ow_device_st dar und zeigt so ein bestimmtes Gert am Bus an.

    \param code One Wire Gertekommando
    \param device Referenz auf das zu benutzende One Wire Gert

    \return Die Funktion ow_function() gibt bei fehlerhafter bertragung eine
    Null(0) zurck. Ansonsten wird eine Eins(1) zurckgegeben. */
extern unsigned char ow_function(unsigned char code, unsigned char device);

#ifdef __cplusplus
}
#endif

/*@}*/

/** \name CodeVision Konformitt */

/*@{*/

#ifdef __cplusplus
extern "C" {
#endif

/** \fn unsigned char w1_init(void)
    \ingroup avrhal_ow

    \codevision

    Aliasfunktion fr ow_init(). Initialisiert die One Wire Gerte am Bus.

    \return Die Funktion w1_init() besitzt den Rckgabewert von ow_init(). */
static inline unsigned char w1_init(void)
{
	return ow_init();
}

/** \fn unsigned char w1_read(void)
    \ingroup avrhal_ow

    \codevision

    Aliasfunktion fr ow_byte_read(). Liest ein Byte vom One Wire Bus.

    \return Die Funktion w1_read() besitzt den Rckgabewert von ow_byte_read(). */
static inline unsigned char w1_read(void)
{
	return ow_byte_read();
}

/** \fn unsigned char w1_write(unsigned char data)
    \ingroup avrhal_ow

    \codevision

    Aliasfunktion fr ow_byte_write(). Schreibt ein Byte zum One Wire Bus.

    \param data zu schreibendes Byte.

    \return Die Funktion w1_write() besitzt den Rckgabewert von
    ow_byte_write(). */
static inline unsigned char w1_write(unsigned char data)
{
	return ow_byte_write(data);
}

/** \fn unsigned char w1_search(unsigned char cmd, void *p)
    \ingroup avrhal_ow

    \codevision

    Aliasfunktion fr ow_rom_search(). Ermittelt die Anzahl der am One Wire
    Bus angeschlossenen Gerte. Wenn kein Gert angeschlossen ist, wird eine
    Null(0) zurckgegeben. Das Byte \em cmd wird in dieser One Wire
    Implementierung ignoriert! Der Zeiger referenziert einen Speicherbereich,
    in dem die 8 Byte breiten ROM Codes der Gerte abgelegt werden. Nach dem
    achten Byte wird ein neuntes Byte fr den Status eines jeden Gertes
    hinterlegt. Daher mu der Anwender 9 Byte fr jedes am One Wire Bus
    angeschlossene Gert bereitstellen.

    \param cmd One Wire Gertekommandobyte (wird ignoriert).
    \param p zeigt auf notwendigen Speicher fr ROM Codes und Statusbytes.

    \return Die Funktion w1_search() besitzt den Rckgabewert von
    ow_rom_search(). */
static inline unsigned char w1_search(
		unsigned char cmd __attribute__ ((unused)),
		void *p)
{
	return ow_rom_search((ow_device_st *)p);
}

/** \fn unsigned char w1_crc8(void *p, unsigned char n)
    \ingroup avrhal_ow

    \codevision

    Aliasfunktion fr crc8_ow(). Ermittelt die im Dallas One Wire Bus verwendete
    8 Bit CRC Summe fr den ber den Zeiger referenzierten Block der angegebenen
    Lnge.

    \param p zeigt auf den Block.
    \param n Gre des Blocks in Byte.

    \return Die Funktion w1_crc8() besitzt den Rckgabewert von crc8_ow(). */
extern unsigned char crc8_ow(unsigned char *, size_t);
static inline unsigned char w1_crc8(void *p, unsigned char n)
{
	return crc8_ow((unsigned char *)p,(size_t)n);
}

#ifdef __cplusplus
}
#endif

/*@}*/

#endif /* __ASSEMBLER__ */

#endif /* _AVRHALL_OW_H_ */
