/*
    Copyright (C) 2003 by Stephan Linz <linz@li-pro.net>

    Some code comes from Brajer Vlado <vlado.brajer@kks.s-net.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Bosten, MA 02111-1307 USA
*/

/* $Id: $ */

#include <avr/io.h>
#include <avrhal/delay.h>
#include <avrhal/lcd.h>

/* lcd_matrix:
 *
 *      Bit:   7 6   5 4 3 2 1 0
 *             | |   \_\_\_\_\_\___ numbers of columns each row
 *             \_\_________________ numbers of rows:
 *                                    00 same as 11 (compatibillity)
 *                                    01 one row
 *                                    10 two rows
 *                                    11 four rows
 */

#define LCD_CMD_DISPLAY_CLEAR		(_BV(0))
#define LCD_CMD_CURSOR_HOME		(_BV(1))
//#define LCD_CMD_ENTRY	... TODO
#define LCD_CMD_DISPLAY_ON		(_BV(3) | _BV(2))
#define LCD_CMD_DISPLAY_OFF		(_BV(3) & ~_BV(2))
#define LCD_CMD_CURSOR_ON		(_BV(3) | _BV(1))
#define LCD_CMD_CURSOR_OFF		(_BV(3) & ~_BV(1))
#define LCD_CMD_BLINKING_ON		(_BV(3) | _BV(0))
#define LCD_CMD_BLINKING_OFF		(_BV(3) & ~_BV(0))
#define LCD_CMD_CGRADDR			(_BV(6))
#define LCD_CMD_DDRADDR			(_BV(7))

/*
 *
 * DD RAM address values for 2x16 sample (ADD):
 *
 *       0  1  2  3  4  5  6  7  8  9  1  1  1  1  1  1|
 *                                     0  1  2  3  4  5|
 *                                                     |
 *  0-  00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F|10 11 12 13 14 15 16...
 *  1-  40 41 42 43 44 45 46 47 48 49 4A 4B 4C 4D 4E 4F|50 51 52 53 54 55 56...
 *       ^
 *       |
 *      _lcd_base_y[(0..3) & _lcd_mask_y] with _lcd_mask_y = 1
 *
 * DD RAM address values for 4x16 sample (ADD):
 *
 *       0  1  2  3  4  5  6  7  8  9  1  1  1  1  1  1|
 *                                     0  1  2  3  4  5|
 *                                                     |
 *  0-  00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F|
 *  1-  40 41 42 43 44 45 46 47 48 49 4A 4B 4C 4D 4E 4F|
 *  2-  10 11 12 13 14 15 16 17 18 19 1A 1B 1C 1D 1E 1F|20 21 22 23 24 25 26...
 *  3-  50 51 52 53 54 55 56 57 58 59 5A 5B 5C 5D 5E 5F|60 61 62 63 64 65 66...
 *       ^
 *       |
 *      _lcd_base_y[(0..3) & _lcd_mask_y] with _lcd_mask_y = 3
 *
 */
#ifdef L_lcd_core
unsigned char _lcd_base_y[4] = { 0x00, 0x40 };
unsigned char _lcd_mask_y, _lcd_x, _lcd_y, _lcd_maxx;
#endif


#define	_lcd_out_p(B,P)							\
	P = (P & _BV(AVRHAL_LIB_LCD_UNUSED_BIT))			\
	  | (B & ~_BV(AVRHAL_LIB_LCD_UNUSED_BIT))

#define _lcd_toggle_EN()						\
	sbi(AVRHAL_LIB_LCD_PORT, AVRHAL_LIB_LCD_EN);	/* EN = 1 */	\
	cbi(AVRHAL_LIB_LCD_PORT, AVRHAL_LIB_LCD_EN);	/* EN = 0 */	

#if AVRHAL_LIB_LCD_COM == LCD_POLLING
#ifdef L_lcd_core
void _lcd_check_BF(void)
{
	// upper nibble as input (temporary)
	_lcd_out_p(0x0f, AVRHAL_LIB_LCD_PORT_DDR);
	sbi(AVRHAL_LIB_LCD_PORT, AVRHAL_LIB_LCD_RD);	// RD = 1
	cbi(AVRHAL_LIB_LCD_PORT, AVRHAL_LIB_LCD_RS);	// RS = 0
	do
	{
		delay_us(1); //delay_6T();
		sbi(AVRHAL_LIB_LCD_PORT, AVRHAL_LIB_LCD_EN);	// EN = 1
		delay_us(1); //delay_6T();
	} while (bit_is_set(AVRHAL_LIB_LCD_PORT_IN, 7));
	cbi(AVRHAL_LIB_LCD_PORT, AVRHAL_LIB_LCD_EN);	// EN = 0
	// back to output port (default mode)
	_lcd_out_p(0xff, AVRHAL_LIB_LCD_PORT_DDR);
}
#endif
#endif

#ifdef L_lcd_core
void _lcd_write_CMD(unsigned char byte)
{
	/* upper nibble */
	_lcd_out_p((byte & 0xf0) & ~_BV(AVRHAL_LIB_LCD_RS)
			& ~_BV(AVRHAL_LIB_LCD_RD), AVRHAL_LIB_LCD_PORT);
	_lcd_toggle_EN();

	/* lower nibble */
	_lcd_out_p((byte << 4) & ~_BV(AVRHAL_LIB_LCD_RS)
			& ~_BV(AVRHAL_LIB_LCD_RD), AVRHAL_LIB_LCD_PORT);
	_lcd_toggle_EN();
}
#endif

#ifdef L_lcd_core
void _lcd_write_DAT(unsigned char byte)
{
	/* upper nibble */
	_lcd_out_p(((byte & 0xf0) | _BV(AVRHAL_LIB_LCD_RS))
			& ~_BV(AVRHAL_LIB_LCD_RD), AVRHAL_LIB_LCD_PORT);
	_lcd_toggle_EN();

	/* lower nibble */
	_lcd_out_p(((byte << 4) | _BV(AVRHAL_LIB_LCD_RS))
			& ~_BV(AVRHAL_LIB_LCD_RD), AVRHAL_LIB_LCD_PORT);
	_lcd_toggle_EN();
}
#endif

#if AVRHAL_LIB_LCD_COM == LCD_POLLING
#ifdef L_lcd_cls
void lcd_cls(void)
{
	_lcd_check_BF();
	_lcd_write_CMD(LCD_CMD_DISPLAY_CLEAR);
}
#endif

#ifdef L_lcd_home
void lcd_home(void)
{
	_lcd_check_BF();
	_lcd_write_CMD(LCD_CMD_CURSOR_HOME);
}
#endif

#ifdef L_lcd_goto
void lcd_goto(unsigned char addr)
{
	_lcd_check_BF();
	_lcd_write_CMD(LCD_CMD_DDRADDR | (addr & 0x7f));
}
#endif

#ifdef L_lcd_write_byte
void lcd_write_byte(unsigned char addr, unsigned char data)
{
	_lcd_check_BF();
	_lcd_write_CMD(addr);
	_lcd_check_BF();
	_lcd_write_DAT(data);
}
#endif
#elif AVRHAL_LIB_LCD_COM == LCD_DELAY
#ifdef L_lcd_cls
void lcd_cls(void)
{
	_lcd_write_CMD(LCD_CMD_DISPLAY_CLEAR);
	delay_us(2000); //delay_ms(2);
}
#endif

#ifdef L_lcd_home
void lcd_home(void)
{
	_lcd_write_CMD(LCD_CMD_CURSOR_HOME);
	delay_us(2000); //delay_ms(2);
}
#endif

#ifdef L_lcd_goto
void lcd_goto(unsigned char addr)
{
	_lcd_write_CMD(LCD_CMD_DDRADDR | (addr & 0x7f));
	delay_us(2000); //delay_ms(2);
}
#endif

#ifdef L_lcd_write_byte
void lcd_write_byte(unsigned char addr, unsigned char data)
{
	_lcd_write_CMD(addr);
	delay_us(50);
	_lcd_write_DAT(data);
	delay_us(50);
}
#endif
#else
#error *** missing or incorrect AVRHAL_LIB_LCD_COM setting
#endif

#ifdef L_lcd_ctrl
void lcd_ctrl(unsigned char lcd_control)
{
	unsigned char temp = 0;

	if (lcd_control & _BV(2)) temp |= LCD_CMD_DISPLAY_ON;
	else temp |= LCD_CMD_DISPLAY_OFF;

	if (lcd_control & _BV(1)) temp |= LCD_CMD_CURSOR_ON;
	else temp |= LCD_CMD_CURSOR_OFF;

	if (lcd_control & _BV(0)) temp |= LCD_CMD_BLINKING_ON;
	else temp |= LCD_CMD_BLINKING_OFF;

#if AVRHAL_LIB_LCD_COM == LCD_POLLING
	_lcd_check_BF();
#endif
	_lcd_write_CMD(temp);
#if AVRHAL_LIB_LCD_COM == LCD_DELAY
	delay_us(50);
#endif

}
#endif

#ifdef L_lcd_gotoxy
void lcd_gotoxy(unsigned char x, unsigned char y)
{
	lcd_goto(_lcd_base_y[(_lcd_y = y & _lcd_mask_y)] + (_lcd_x = x));
}
#endif

#ifdef L_lcd_core
void lcd_init(unsigned char lcd_matrix)
{
	_lcd_maxx	= lcd_matrix & 0x3f;	// save for further usage
	_lcd_base_y[2]	= _lcd_maxx;
	_lcd_base_y[3]	= _lcd_maxx + 0x40;

	if ((lcd_matrix >> 6) == 0 || (lcd_matrix >> 6) == 3)
	{
		_lcd_mask_y = 3;
	}
	else
	{
		_lcd_mask_y = lcd_matrix >> 7;
	}

	/* output port (default mode) */
	_lcd_out_p(0xff, AVRHAL_LIB_LCD_PORT_DDR);

	/* avoid power-on mismatches at LCD device */
	delay_us(20000); //delay_ms(20);

	_lcd_out_p(0x30, AVRHAL_LIB_LCD_PORT);	/* init step 1		*/
	_lcd_toggle_EN(); delay_us(5000); //delay_ms(5);
	_lcd_out_p(0x30, AVRHAL_LIB_LCD_PORT);	/* init step 2		*/
	_lcd_toggle_EN(); delay_us(110);
	_lcd_out_p(0x30, AVRHAL_LIB_LCD_PORT);	/* init step 3		*/
	_lcd_toggle_EN(); delay_us(50);
	_lcd_out_p(0x20, AVRHAL_LIB_LCD_PORT);	/* init 4bit mode	*/
	_lcd_toggle_EN(); delay_us(50);
	_lcd_out_p(0x20, AVRHAL_LIB_LCD_PORT);	/* functionset		*/
	_lcd_toggle_EN();
	_lcd_out_p(0x80, AVRHAL_LIB_LCD_PORT);	/* 4bit, dual line	*/
	_lcd_toggle_EN(); delay_us(50);
	_lcd_out_p(0x00, AVRHAL_LIB_LCD_PORT);	/* entry mode:		*/
	_lcd_toggle_EN();
	_lcd_out_p(0x60, AVRHAL_LIB_LCD_PORT);	/* increment cursor	*/
	_lcd_toggle_EN(); delay_us(50);

	lcd_cls();		/* be shure, there is nothing displayed */
	lcd_home();
	lcd_ctrl(0x07);		/* display, cursor, and blink on	*/
}
#endif

#ifdef L_lcd_putch
void lcd_putch(unsigned char data)
{
#if AVRHAL_LIB_LCD_COM == LCD_POLLING
	_lcd_check_BF();
#endif
	_lcd_write_DAT(data);
#if AVRHAL_LIB_LCD_COM == LCD_DELAY
	delay_us(50);
#endif
	if (++_lcd_x == _lcd_maxx) lcd_gotoxy(0, _lcd_y + 1);
}
#endif

#ifdef L_lcd_cleol
void lcd_cleol(void)
{
	unsigned char _lcd_x_save = _lcd_x;
	do lcd_putch(' '); while (_lcd_x != 0);
	lcd_gotoxy(_lcd_x_save, _lcd_y - 1);
}
#endif

#ifdef L_lcd_putstr
void lcd_putstr(unsigned char * data)
{
	while(*data) {
		lcd_putch(*data);
		data++;
	}
}
#endif
