/*
    Copyright (C) 2003 by Stephan Linz <linz@li-pro.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Bosten, MA 02111-1307 USA
*/

/* $Id: $ */

#include <avr/io.h>
#include <avrhal/led.h>

/* led_options:
 *
 * Bit:   7 6 5 4   3 2 1   0
 *        | | | |   | | |   \___ fill mode (bar or point graph)
 *        | | | |   | | |           0 point graph
 *        | | | |   | | |           1 bar graph
 *        | | | |   \_\_\_______ margin bit position
 *        | | | |                 000 Bit 0 (bottom)
 *        | | | |                 100 Bit 4 (middle)
 *        | | | |                 111 Bit 7 (top)
 */
static unsigned char	_led_opt = LED_BAR | LED_MARGIN(0);
static unsigned int	_led_qnt = 0;

static void _led_quant(int led_maxlevel)
{
	if (led_maxlevel > 8)
	{
		_led_qnt = led_maxlevel / 8;
#if 0
		_led_opt = LED_BAR | LED_MARGIN(/* Bit */ 0 /* is margin */);
#endif
	}
	else if (led_maxlevel < -8)
	{
		_led_qnt = led_maxlevel / -8;
#if 0
		_led_opt = LED_BAR | LED_MARGIN(/* Bit */ 7 /* is margin */);
#endif
	}
	else
	{
		_led_qnt = 0;
		/* margin or other led_options are unused - raw i/f */
	}
}

static void _led_graph(signed char bit)
{
	signed char absbit = bit + ((_led_opt >> 1) & 0x7);

	if (_led_opt & LED_BAR)
	{
		if (absbit > (absbit - bit))	/* bar from bottom to top */
		{
			AVRHAL_LIB_LED_PORT = (((1 << absbit) + ((1 << absbit) - 1))
					& ~((1 << (absbit - bit)) - 1));
			return;
		}
		else if (absbit < (absbit - bit))	/* bar from top to bottom */
		{
			AVRHAL_LIB_LED_PORT = (((1 << (absbit - bit)) - 1) & ~((1 << absbit) - 1));
			return;
		}
	}

	if (absbit < 0)
	{
		AVRHAL_LIB_LED_PORT = AVRHAL_LIB_LED_OFF;
		return;
	}

	AVRHAL_LIB_LED_PORT = (1 << absbit);
}

void led_graph_put(int bitset_level)
{
	if (_led_qnt)
		/* use led_options for output */
		_led_graph((signed char)(bitset_level / _led_qnt));
	else
		/* raw i/f */
		AVRHAL_LIB_LED_PORT = (unsigned char)bitset_level;
}

void led_graph_control(int led_maxlevel, signed char led_options)
{
	_led_quant(led_maxlevel);
	_led_opt = led_options;
}
